# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

"""
Forked from https://projects.blender.org/blender/blender-addons/src/branch/main/io_scene_3ds

This code exports to a new .p2 format; both extending and incompatible with, the .3ds format.

This divergence is necessary to support:

    - exporting multiple UV coords per vertex,
    - exporting animation transforms for each frame, even if keyframeless,
    - exporting custom animation Vertex Groups (bones), each with differing bone weights,
    - handling multiple 16-bit vertex flags per vertex (e.g. when vertex is linked to >1 Vertex Group),
    - several other small changes too numerous to list here
"""

import bpy
import math
import struct
import mathutils
import bpy_extras
from bpy_extras import node_shader_utils

SZ_SHORT            = 2
SZ_INT              = 4
SZ_FLOAT            = 4

RGB                 = 0x0010  # RGB float
RGB1                = 0x0011  # RGB Color1
RGB2                = 0x0012  # RGB Color2
LIN_COLOR_F         = 0x0013  # Linear colour (3 floats)
PCT                 = 0x0030  # Percent (short integer)
PCTF                = 0x0031  # Percent (32-bit float)
MASTERSCALE         = 0x0100  # Master scale factor

# ----- Identifier Chunk, at the beginning of each file
M3DMAGIC            = 0x4D4D

# ------ Main Chunks
M3D_VERSION         = 0x0002  # Specifies the file version
MDATA               = 0x3D3D  # Main mesh object chunk
KFDATA              = 0xB000  # Keyframe data header

# ------ sub defines of MDATA
BITMAP              = 0x1100  # The background image name
USE_BITMAP          = 0x1101  # The background image flag
SOLIDBACKGND        = 0x1200  # The background color (RGB)
USE_SOLIDBGND       = 0x1201  # The background color flag
VGRADIENT           = 0x1300  # The background gradient colors
USE_VGRADIENT       = 0x1301  # The background gradient flag
O_CONSTS            = 0x1500  # The origin of the 3D cursor
AMBIENT_LIGHT       = 0x2100  # Ambient light colour data
FOG                 = 0x2200  # The fog atmosphere settings
USE_FOG             = 0x2201  # The fog atmosphere flag
LAYER_FOG           = 0x2302  # The fog layer atmosphere settings
USE_LAYER_FOG       = 0x2303  # The fog layer atmosphere flag
MESH_VERSION        = 0x3D3E  # Version number of the mesh
NAMED_OBJECT        = 0x4000  # Faces, vertices, lights, cameras
MAT_ENTRY           = 0xAFFF  # Material (i.e. texture) infomation

# >------ sub defines of NAMED_OBJECT
OBJECT_MESH         = 0x4100  # Mesh object
OBJECT_LIGHT        = 0x4600  # Light object
OBJECT_CAMERA       = 0x4700  # Camera object
OBJECT_HIERARCHY    = 0x4F00  # Hierarchy id of the object
OBJECT_PARENT       = 0x4F10  # Parent id of the object

# >------ sub defines of OBJECT_MESH
OBJECT_VERTICES     = 0x4110  # The objects vertices
OBJECT_VERTFLAG     = 0x4111  # Vertex flags (*** Note: These are custom encoded 32-bit flags ***)
OBJECT_POSEBONES    = 0x4112  # Armature Pose Bones (*** Note: This is a non-3DS custom chunk ID ***)
OBJECT_FACES        = 0x4120  # The objects faces
OBJECT_MATERIAL     = 0x4130  # This is found if the object has a material, either texture map or color
OBJECT_UV           = 0x4140  # The UV texture coordinates
OBJECT_SMOOTH       = 0x4150  # The objects smooth groups
OBJECT_MESH_MATRIX  = 0x4160  # The objects transformation matrix (location and rotation)

# >------ Sub defines of OBJECT_LIGHT
LIGHT_SPOTLIGHT     = 0x4610  # The target of a spotlight
LIGHT_ATTENUATE     = 0x4625  # Light attenuation flag
LIGHT_SPOT_SHADOWED = 0x4630  # Light spot shadow flag
LIGHT_SPOT_LSHADOW  = 0x4641  # Light spot shadow parameters
LIGHT_SPOT_SEE_CONE = 0x4650  # Light spot show cone flag
LIGHT_SPOT_RECTANGLE= 0x4651  # Light spot rectangle flag
LIGHT_SPOT_OVERSHOOT= 0x4652  # Light spot overshoot flag
LIGHT_SPOT_PROJECTOR= 0x4653  # Light spot projection bitmap
LIGHT_SPOT_ROLL     = 0x4656  # Light spot roll angle
LIGHT_SPOT_ASPECT   = 0x4657  # Light spot aspect ratio
LIGHT_INNER_RANGE   = 0x4659  # Light inner range value
LIGHT_OUTER_RANGE   = 0x465A  # Light outer range value
LIGHT_MULTIPLIER    = 0x465B  # The light energy factor

# >------ sub defines of OBJECT_CAMERA
OBJECT_CAM_RANGES   = 0x4720  # The camera range values

# ------ sub defines of MAT_ENTRY
MATNAME             = 0xA000  # This holds the material name
MATAMBIENT          = 0xA010  # Ambient color of the object/material
MATDIFFUSE          = 0xA020  # This holds the color of the object/material
MATSPECULAR         = 0xA030  # Specular color of the object/material
MATSHINESS          = 0xA040  # Specular intensity of the object/material (percent)
MATSHIN2            = 0xA041  # Reflection of the object/material (percent)
MATSHIN3            = 0xA042  # metallic/mirror of the object/material (percent)
MATTRANS            = 0xA050  # Transparency value (100-OpacityValue) (percent)
MATSELFILLUM        = 0xA080  # # Material self illumination flag
MATSELFILPCT        = 0xA084  # Self illumination strength (percent)
MATWIRE             = 0xA085  # Material wireframe rendered flag
MATFACEMAP          = 0xA088  # Face mapped textures flag
MATPHONGSOFT        = 0xA08C  # Phong soften material flag
MATWIREABS          = 0xA08E  # Wire size in units flag
MATWIRESIZE         = 0xA087  # Rendered wire size in pixels
MATSHADING          = 0xA100  # Material shading method
MAT_DIFFUSEMAP      = 0xA200  # Header for a new diffuse texture
MAT_SPECMAP         = 0xA204  # head for specularity map
MAT_OPACMAP         = 0xA210  # head for opacity map
MAT_REFLMAP         = 0xA220  # head for reflect map
MAT_BUMPMAP         = 0xA230  # head for normal map
MAT_BUMPPERCENT     = 0xA252  # Normalmap strength (percent)
MAT_TEX2MAP         = 0xA33A  # head for secondary texture
MAT_SHINMAP         = 0xA33C  # head for roughness map
MAT_SELFIMAP        = 0xA33D  # head for emission map
MATMAPFILE          = 0xA300  # This holds the file name of a texture
MAT_MAP_TILING      = 0xA351  # 2nd bit (from LSB) is mirror UV flag
MAT_MAP_TEXBLUR     = 0xA353  # Texture blurring factor
MAT_MAP_USCALE      = 0xA354  # U axis scaling
MAT_MAP_VSCALE      = 0xA356  # V axis scaling
MAT_MAP_UOFFSET     = 0xA358  # U axis offset
MAT_MAP_VOFFSET     = 0xA35A  # V axis offset
MAT_MAP_ANG         = 0xA35C  # UV rotation around the z-axis in rad
MAP_COL1            = 0xA360  # Tint Color1
MAP_COL2            = 0xA362  # Tint Color2
MAP_RCOL            = 0xA364  # Red tint
MAP_GCOL            = 0xA366  # Green tint
MAP_BCOL            = 0xA368  # Blue tint

# >------ sub defines of KFDATA
AMBIENT_NODE_TAG    = 0xB001  # Ambient node tag
OBJECT_NODE_TAG     = 0xB002  # Object node tag
CAMERA_NODE_TAG     = 0xB003  # Camera node tag
TARGET_NODE_TAG     = 0xB004  # Target node tag
LIGHT_NODE_TAG      = 0xB005  # Light node tag
LTARGET_NODE_TAG    = 0xB006  # Light target tag
SPOT_NODE_TAG       = 0xB007  # Spotlight tag
KFDATA_KFSEG        = 0xB008  # Frame start & end
KFDATA_KFCURTIME    = 0xB009  # Frame current time
KFDATA_KFHDR        = 0xB00A  # Keyframe header
OBJECT_NODE_HDR     = 0xB010  # Hierachy tree header
OBJECT_INSTANCENAME = 0xB011  # Object instance name
OBJECT_PIVOT        = 0xB013  # Object pivot position
OBJECT_BOUNDBOX     = 0xB014  # Object boundbox
OBJECT_MORPH_SMOOTH = 0xB015  # Object smooth angle
POS_TRACK_TAG       = 0xB020  # Position transform tag
ROT_TRACK_TAG       = 0xB021  # Rotation transform tag
SCL_TRACK_TAG       = 0xB022  # Scale transform tag
FOV_TRACK_TAG       = 0xB023  # Field of view tag
ROLL_TRACK_TAG      = 0xB024  # Roll transform tag
COL_TRACK_TAG       = 0xB025  # Color transform tag
HOTSPOT_TRACK_TAG   = 0xB027  # Hotspot transform tag
FALLOFF_TRACK_TAG   = 0xB028  # Falloff transform tag
OBJECT_NODE_ID      = 0xB030  # Object hierachy ID

OBJECT_PARENT_NAME  = 0x80F0  # Object parent name tree

ROOT_OBJECT         = 0xFFFF  # Root object

name_unique         = []      # stores string, ascii only
name_mapping        = {}      # stores {orig: byte} mapping

def sane_name(name):
    name_fixed = name_mapping.get(name)
    if name_fixed is not None:
        return name_fixed

    # strip non ascii chars
    new_name_clean = new_name = name.encode("ASCII", "replace").decode("ASCII")[:63]
    i = 0

    while new_name in name_unique:
        new_name = new_name_clean + ".%.3d" % i
        i += 1

    # note, appending the 'str' version.
    name_unique.append(new_name)
    name_mapping[name] = new_name = new_name.encode("ASCII", "replace")
    return new_name


def uv_key(uv):
    return round(uv[0], 6), round(uv[1], 6)


class _3ds_ushort(object):
    
    """Class representing a short (2-byte integer) for a P2 file."""
    
    __slots__ = ("value", )

    def __init__(self, val=0):
        self.value = val

    def get_size(self):
        return SZ_SHORT

    def write(self, file):
        file.write(struct.pack("<H", self.value))

    def __str__(self):
        return str(self.value)


class _3ds_uint(object):
    
    """Class representing an int (4-byte integer) for a P2 file."""
    
    __slots__ = ("value", )

    def __init__(self, val):
        self.value = val

    def get_size(self):
        return SZ_INT

    def write(self, file):
        file.write(struct.pack("<I", self.value))

    def __str__(self):
        return str(self.value)


class _3ds_float(object):
    
    """Class representing a 4-byte IEEE floating point number for a P2 file."""
    
    __slots__ = ("value", )

    def __init__(self, val):
        self.value = val

    def get_size(self):
        return SZ_FLOAT

    def write(self, file):
        file.write(struct.pack("<f", self.value))

    def __str__(self):
        return str(self.value)


class _3ds_string(object):
    
    """Class representing a zero-terminated string for a filename."""
    
    __slots__ = ("value", )

    def __init__(self, val):
        assert(type(val) == bytes)
        self.value = val

    def get_size(self):
        return (len(self.value) + 1)

    def write(self, file):
        binary_format = "<%ds" % (len(self.value) + 1)
        file.write(struct.pack(binary_format, self.value))

    def __str__(self):
        return str(self.value)


class _3ds_packed_string(object):
    
    """Class representing a packed (NON zero-terminated) string of 'bytes' type for a P2 file."""
    
    __slots__ = ("value", )

    def __init__(self, val):
        assert(type(val) == bytes)
        self.value = val

    def get_size(self):
        return (len(self.value))

    def write(self, file):
        binary_format = "<%ds" % (len(self.value))
        file.write(struct.pack(binary_format, self.value))

    def __str__(self):
        return str(self.value)


class _3ds_point_3d(object):
    
    """Class representing a three-dimensional point for a P2 file."""
    
    __slots__ = "x", "y", "z"

    def __init__(self, point):
        self.x, self.y, self.z = point

    def get_size(self):
        return 3 * SZ_FLOAT

    def write(self, file):
        file.write(struct.pack('<3f', self.x, self.y, self.z))

    def __str__(self):
        return '(%f, %f, %f)' % (self.x, self.y, self.z)


class _3ds_point_4d(object):

    """Class representing a four-dimensional point for a P2 file, for instance a quaternion."""
    
    __slots__ = "x","y","z","w"
    def __init__(self, point=(0.0,0.0,0.0,0.0)):
        self.x, self.y, self.z, self.w = point

    def get_size(self):
        return 4*SZ_FLOAT

    def write(self,file):
        data=struct.pack('<4f', self.x, self.y, self.z, self.w)
        file.write(data)

    def __str__(self):
        return '(%f, %f, %f, %f)' % (self.x, self.y, self.z, self.w)


class _3ds_point_uv(object):
    
    """Class representing a UV-coordinate for a P2 file."""
    
    __slots__ = ("uv", )

    def __init__(self, point):
        self.uv = point

    def get_size(self):
        return 2 * SZ_FLOAT

    def write(self, file):
        data = struct.pack('<2f', self.uv[0], self.uv[1])
        file.write(data)

    def __str__(self):
        return '(%g, %g)' % self.uv


class _3ds_float_color(object):
    
    """Class representing a rgb float color for a P2 file."""
    
    __slots__ = "r", "g", "b"

    def __init__(self, col):
        self.r, self.g, self.b = col

    def get_size(self):
        return 3 * SZ_FLOAT

    def write(self, file):
        file.write(struct.pack('3f', self.r, self.g, self.b))

    def __str__(self):
        return '{%f, %f, %f}' % (self.r, self.g, self.b)


class _3ds_rgb_color(object):
    
    """Class representing a (24-bit) rgb color for a P2 file."""
    
    __slots__ = "r", "g", "b"

    def __init__(self, col):
        self.r, self.g, self.b = col

    def get_size(self):
        return 3

    def write(self, file):
        file.write(struct.pack('<3B', int(255 * self.r), int(255 * self.g), int(255 * self.b)))

    def __str__(self):
        return '{%f, %f, %f}' % (self.r, self.g, self.b)


class _3ds_face(object):
    
    """Class representing a face for a P2 file."""
    
    __slots__ = ("vindex", )

    def __init__(self, vindex):
        self.vindex = vindex

    def get_size(self):
        return 4 * SZ_SHORT

    # no need to validate every face vert. the oversized array check will catch this problem

    def write(self, file):
        # The last zero is only used by 3d studio
        file.write(struct.pack("<4H", self.vindex[0], self.vindex[1], self.vindex[2], 0))

    def __str__(self):
        return "[%d %d %d]" % (self.vindex[0], self.vindex[1], self.vindex[2])


class _3ds_array(object):
    
    """Class representing an array of variables for a P2 file.

    Consists of a _3ds_ushort to indicate the number of items, followed by the items themselves."""
    
    __slots__ = "values", "size"

    def __init__(self):
        self.values = []
        self.size = SZ_SHORT

    # add an item:
    def add(self, item):
        self.values.append(item)
        self.size += item.get_size()

    def get_size(self):
        return self.size

    def validate(self):
        return len(self.values) <= 65535

    def write(self, file):
        _3ds_ushort(len(self.values)).write(file)
        for value in self.values:
            value.write(file)

    # To not overwhelm the output in a dump, a _3ds_array only
    # outputs the number of items, not all of the actual items.
    def __str__(self):
        return '(%d items)' % len(self.values)


class _3ds_uint_array(object):
    
    """Class representing an array of variables for a P2 file.

    Consists of a _3ds_uint to indicate the number of items, followed by the items themselves."""
    
    __slots__ = "values", "size"

    def __init__(self):
        self.values = []
        self.size = SZ_INT

    # add an item:
    def add(self, item):
        self.values.append(item)
        self.size += item.get_size()

    def get_size(self):
        return self.size
    
    def write(self, file):
        _3ds_uint(len(self.values)).write(file)
        for value in self.values:
            value.write(file)

    # To not overwhelm the output in a dump, a _3ds_uint_array only
    # outputs the number of items, not all of the actual items.
    def __str__(self):
        return '(%d items)' % len(self.values)

class _3ds_string_array(object):
    
    """Class representing an array of _3ds_string variables.

    Consists of a _3ds_ushort to indicate the number of items, followed by _3ds_string items."""
    
    __slots__ = "values", "size"

    def __init__(self):
        self.values = []
        self.size = SZ_SHORT

    # add an item:
    def add(self, item):
        self.values.append(item)
        self.size += item.get_size()

    def get_size(self):
        return self.size

    def validate(self):
        return len(self.values) <= 65535

    def write(self, file):
        _3ds_ushort(len(self.values)).write(file)
        for value in self.values:
            value.write(file)

    # To not overwhelm the output in a dump, a _3ds_string_array only
    # outputs the number of items, not all of the actual items.
    def __str__(self):
        return '(%d items)' % len(self.values)


class _3ds_named_variable(object):
    
    """Convenience class for named variables."""

    __slots__ = "value", "name"

    def __init__(self, name, val=None):
        self.name = name
        self.value = val

    def get_size(self):
        if self.value is None:
            return 0
        else:
            return self.value.get_size()

    def write(self, file):
        if self.value is not None:
            self.value.write(file)

    def dump(self, indent):
        if self.value is not None:
            print(indent * " ",
                  self.name if self.name else "[unnamed]",
                  " = ",
                  self.value)


# the chunk class
class _3ds_chunk(object):
    
    """Class representing a chunk in a P2 file.

    Chunks contain zero or more variables, followed by zero or more subchunks."""
    
    __slots__ = "ID", "size", "variables", "subchunks"

    def __init__(self, chunk_id=0):
        self.ID = _3ds_ushort(chunk_id)
        self.size = _3ds_uint(0)
        self.variables = []
        self.subchunks = []

    def add_variable(self, name, var):
        
        """Add a named variable. The name is mostly for debugging purposes."""
        
        self.variables.append(_3ds_named_variable(name, var))

    def add_subchunk(self, chunk):
        
        """Add a subchunk."""
        
        self.subchunks.append(chunk)

    def get_size(self):
        
        """Calculate the size of the chunk and return it.

        The sizes of the variables and subchunks are used to determine this chunk\'s size."""
        
        tmpsize = self.ID.get_size() + self.size.get_size()
        for variable in self.variables:
            tmpsize += variable.get_size()
        for subchunk in self.subchunks:
            tmpsize += subchunk.get_size()
        self.size.value = tmpsize
        return self.size.value

    def validate(self):
        
        for var in self.variables:
            func = getattr(var.value, "validate", None)
            if (func is not None) and not func():
                return False

        for chunk in self.subchunks:
            func = getattr(chunk, "validate", None)
            if (func is not None) and not func():
                return False

        return True

    def write(self, file):
        
        """Write the chunk to a file.

        Uses the write function of the variables and the subchunks to do the actual work."""
        
        # write header
        self.ID.write(file)
        self.size.write(file)
        
        for variable in self.variables:
            variable.write(file)
        for subchunk in self.subchunks:
            subchunk.write(file)

    def dump(self, indent=0):
        
        """Write the chunk to a file.

        Dump is used for debugging purposes, to dump the contents of a chunk to the standard output.
        Uses the dump function of the named variables and the subchunks to do the actual work."""
        
        print(indent * " ", "ID=%r" % hex(self.ID.value), "size=%r" % self.get_size())
        
        for variable in self.variables:
            variable.dump(indent + 1)
        for subchunk in self.subchunks:
            subchunk.dump(indent + 1)


def get_material_image(material):
    
    """ Get images from paint slots."""
    
    if material:
        pt = material.paint_active_slot
        tex = material.texture_paint_images
        if pt < len(tex):
            slot = tex[pt]
            if slot.type == 'IMAGE':
                return slot


def get_uv_image(ma):
    
    """ Get image from material wrapper."""
    
    if ma and ma.use_nodes:
        ma_wrap = node_shader_utils.PrincipledBSDFWrapper(ma)
        ma_tex = ma_wrap.base_color_texture
        if ma_tex and ma_tex.image is not None:
            return ma_tex.image
    else:
        return get_material_image(ma)


def make_material_subchunk(chunk_id, color):
    
    """Make a material subchunk.

    Used for color subchunks, such as diffuse color or ambient color subchunks."""
    
    mat_sub = _3ds_chunk(chunk_id)
    col1 = _3ds_chunk(RGB1)
    col1.add_variable("color1", _3ds_rgb_color(color))
    mat_sub.add_subchunk(col1)
    # optional:
    #col2 = _3ds_chunk(RGB1)
    #col2.add_variable("color2", _3ds_rgb_color(color))
    # mat_sub.add_subchunk(col2)
    return mat_sub


def make_percent_subchunk(chunk_id, percent):
    
    """Make a percentage based subchunk."""
    
    pct_sub = _3ds_chunk(chunk_id)
    pcti = _3ds_chunk(PCT)
    pcti.add_variable("percent", _3ds_ushort(int(round(percent * 100, 0))))
    pct_sub.add_subchunk(pcti)
    return pct_sub


def make_texture_chunk(chunk_id, images):
    
    """Make Material Map texture chunk."""
    
    # Add texture percentage value (100 = 1.0)
    ma_sub = make_percent_subchunk(chunk_id, 1)
    has_entry = False

    def add_image(img):
        filename = bpy.path.basename(image.filepath)
        ma_sub_file = _3ds_chunk(MATMAPFILE)
        ma_sub_file.add_variable("image", _3ds_string(sane_name(filename)))
        ma_sub.add_subchunk(ma_sub_file)

    for image in images:
        add_image(image)
        has_entry = True

    return ma_sub if has_entry else None


def make_material_texture_chunk(chunk_id, texslots, pct):
    
    """Make Material Map texture chunk given a seq. of `MaterialTextureSlot`'s
        Paint slots are optionally used as image source if no nodes are
        used. No additional filtering for mapping modes is done, all
        slots are written "as is"."""
        
    # Add texture percentage value
    mat_sub = make_percent_subchunk(chunk_id, pct)
    has_entry = False

    def add_texslot(texslot):
        image = texslot.image

        filename = bpy.path.basename(image.filepath)
        mat_sub_file = _3ds_chunk(MATMAPFILE)
        mat_sub_file.add_variable("mapfile", _3ds_string(sane_name(filename)))
        mat_sub.add_subchunk(mat_sub_file)
        for link in texslot.socket_dst.links:
            socket = link.from_socket.identifier

        maptile = 0

        # no perfect mapping for mirror modes - 3DS only has uniform mirror w. repeat=2
        if texslot.extension == 'EXTEND':
            maptile |= 0x1
        # CLIP maps to 3DS' decal flag
        elif texslot.extension == 'CLIP':
            maptile |= 0x10

        mat_sub_tile = _3ds_chunk(MAT_MAP_TILING)
        mat_sub_tile.add_variable("tiling", _3ds_ushort(maptile))
        mat_sub.add_subchunk(mat_sub_tile)

        if socket == 'Alpha':
            mat_sub_alpha = _3ds_chunk(MAT_MAP_TILING)
            alphaflag = 0x40  # summed area sampling 0x20
            mat_sub_alpha.add_variable("alpha", _3ds_ushort(alphaflag))
            mat_sub.add_subchunk(mat_sub_alpha)
            if texslot.socket_dst.identifier in {'Base Color', 'Specular'}:
                mat_sub_tint = _3ds_chunk(MAT_MAP_TILING)  # RGB tint 0x200
                tint = 0x80 if texslot.image.colorspace_settings.name == 'Non-Color' else 0x200
                mat_sub_tint.add_variable("tint", _3ds_ushort(tint))
                mat_sub.add_subchunk(mat_sub_tint)

        mat_sub_texblur = _3ds_chunk(MAT_MAP_TEXBLUR)  # Based on observation this is usually 1.0
        mat_sub_texblur.add_variable("maptexblur", _3ds_float(1.0))
        mat_sub.add_subchunk(mat_sub_texblur)

        mat_sub_uscale = _3ds_chunk(MAT_MAP_USCALE)
        mat_sub_uscale.add_variable("mapuscale", _3ds_float(round(texslot.scale[0], 6)))
        mat_sub.add_subchunk(mat_sub_uscale)

        mat_sub_vscale = _3ds_chunk(MAT_MAP_VSCALE)
        mat_sub_vscale.add_variable("mapvscale", _3ds_float(round(texslot.scale[1], 6)))
        mat_sub.add_subchunk(mat_sub_vscale)

        mat_sub_uoffset = _3ds_chunk(MAT_MAP_UOFFSET)
        mat_sub_uoffset.add_variable("mapuoffset", _3ds_float(round(texslot.translation[0], 6)))
        mat_sub.add_subchunk(mat_sub_uoffset)

        mat_sub_voffset = _3ds_chunk(MAT_MAP_VOFFSET)
        mat_sub_voffset.add_variable("mapvoffset", _3ds_float(round(texslot.translation[1], 6)))
        mat_sub.add_subchunk(mat_sub_voffset)

        mat_sub_angle = _3ds_chunk(MAT_MAP_ANG)
        mat_sub_angle.add_variable("mapangle", _3ds_float(round(texslot.rotation[2], 6)))
        mat_sub.add_subchunk(mat_sub_angle)

        if texslot.socket_dst.identifier in {'Base Color', 'Specular'}:
            rgb = _3ds_chunk(MAP_COL1)  # Add tint color
            base = texslot.owner_shader.material.diffuse_color[:3]
            spec = texslot.owner_shader.material.specular_color[:]
            rgb.add_variable("mapcolor", _3ds_rgb_color(spec if texslot.socket_dst.identifier == 'Specular' else base))
            mat_sub.add_subchunk(rgb)

    # store all textures for this mapto in order. This at least is what
    # the 3DS exporter did so far, afaik most readers will just skip
    # over 2nd textures.
    for slot in texslots:
        if slot.image is not None:
            add_texslot(slot)
            has_entry = True

    return mat_sub if has_entry else None


def make_material_chunk(material, image):
    
    """Make a material chunk out of a blender material.
    Shading method is required for 3ds max, 0 for wireframe.
    0x1 for flat, 0x2 for gouraud, 0x3 for phong and 0x4 for metal."""
    
    material_chunk = _3ds_chunk(MAT_ENTRY)
    name = _3ds_chunk(MATNAME)
    shading = _3ds_chunk(MATSHADING)

    name_str = material.name if material else "None"

    if image:
        name_str += image.name

    name.add_variable("name", _3ds_string(sane_name(name_str)))
    material_chunk.add_subchunk(name)

    if not material:
        shading.add_variable("shading", _3ds_ushort(1))  # Flat shading
        material_chunk.add_subchunk(make_material_subchunk(MATAMBIENT, (0.0, 0.0, 0.0)))
        material_chunk.add_subchunk(make_material_subchunk(MATDIFFUSE, (0.8, 0.8, 0.8)))
        material_chunk.add_subchunk(make_material_subchunk(MATSPECULAR, (1.0, 1.0, 1.0)))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHINESS, .2))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHIN2, 1))
        material_chunk.add_subchunk(shading)

    elif material and material.use_nodes:
        wrap = node_shader_utils.PrincipledBSDFWrapper(material)
        shading.add_variable("shading", _3ds_ushort(3))  # Phong shading
        material_chunk.add_subchunk(make_material_subchunk(MATAMBIENT, wrap.emission_color[:3]))
        material_chunk.add_subchunk(make_material_subchunk(MATDIFFUSE, wrap.base_color[:3]))
        material_chunk.add_subchunk(make_material_subchunk(MATSPECULAR, material.specular_color[:]))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHINESS, wrap.roughness))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHIN2, wrap.specular))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHIN3, wrap.metallic))
        material_chunk.add_subchunk(make_percent_subchunk(MATTRANS, 1 - wrap.alpha))
        material_chunk.add_subchunk(shading)

        if wrap.base_color_texture:
            d_pct = 0.7 + sum(wrap.base_color[:]) * 0.1
            color = [wrap.base_color_texture]
            matmap = make_material_texture_chunk(MAT_DIFFUSEMAP, color, d_pct)
            if matmap:
                material_chunk.add_subchunk(matmap)

        if wrap.specular_texture:
            spec = [wrap.specular_texture]
            s_pct = material.specular_intensity
            matmap = make_material_texture_chunk(MAT_SPECMAP, spec, s_pct)
            if matmap:
                material_chunk.add_subchunk(matmap)

        if wrap.alpha_texture:
            alpha = [wrap.alpha_texture]
            a_pct = material.diffuse_color[3]
            matmap = make_material_texture_chunk(MAT_OPACMAP, alpha, a_pct)
            if matmap:
                material_chunk.add_subchunk(matmap)

        if wrap.metallic_texture:
            metallic = [wrap.metallic_texture]
            m_pct = material.metallic
            matmap = make_material_texture_chunk(MAT_REFLMAP, metallic, m_pct)
            if matmap:
                material_chunk.add_subchunk(matmap)

        if wrap.normalmap_texture:
            normal = [wrap.normalmap_texture]
            bump = wrap.normalmap_strength
            b_pct = min(bump, 1)
            bumpval = min(999, (bump * 100))  # 3ds max bump = 999
            strength = _3ds_chunk(MAT_BUMPPERCENT)
            strength.add_variable("bump_pct", _3ds_ushort(int(bumpval)))
            matmap = make_material_texture_chunk(MAT_BUMPMAP, normal, b_pct)
            if matmap:
                material_chunk.add_subchunk(matmap)
                material_chunk.add_subchunk(strength)

        if wrap.roughness_texture:
            roughness = [wrap.roughness_texture]
            r_pct = material.roughness
            matmap = make_material_texture_chunk(MAT_SHINMAP, roughness, r_pct)
            if matmap:
                material_chunk.add_subchunk(matmap)

        if wrap.emission_color_texture:
            e_pct = sum(wrap.emission_color[:]) * .25
            emission = [wrap.emission_color_texture]
            matmap = make_material_texture_chunk(MAT_SELFIMAP, emission, e_pct)
            if matmap:
                material_chunk.add_subchunk(matmap)

        # make sure no textures are lost. Everything that doesn't fit
        # into a channel is exported as secondary texture
        secondaryTexture = []

        for link in wrap.material.node_tree.links:
            if link.from_node.type == 'TEX_IMAGE' and link.to_node.type != 'BSDF_PRINCIPLED':
                secondaryTexture = [link.from_node.image] if not wrap.normalmap_texture else None

        if secondaryTexture:
            matmap = make_texture_chunk(MAT_TEX2MAP, secondaryTexture)
            if matmap:
                material_chunk.add_subchunk(matmap)

    else:
        shading.add_variable("shading", _3ds_ushort(2))  # Gouraud shading
        material_chunk.add_subchunk(make_material_subchunk(MATAMBIENT, material.line_color[:3]))
        material_chunk.add_subchunk(make_material_subchunk(MATDIFFUSE, material.diffuse_color[:3]))
        material_chunk.add_subchunk(make_material_subchunk(MATSPECULAR, material.specular_color[:]))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHINESS, material.roughness))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHIN2, material.specular_intensity))
        material_chunk.add_subchunk(make_percent_subchunk(MATSHIN3, material.metallic))
        material_chunk.add_subchunk(make_percent_subchunk(MATTRANS, 1 - material.diffuse_color[3]))
        material_chunk.add_subchunk(shading)

        slots = [get_material_image(material)]  # can be None

        if image:
            material_chunk.add_subchunk(make_texture_chunk(MAT_DIFFUSEMAP, slots))

    return material_chunk


class tri_wrapper(object):
    
    """Class representing a triangle.

    Used when converting faces to triangles"""

    __slots__ = "vertex_index", "ma", "image", "faceuvs", "offset", "group"

    def __init__(self, vindex=(0, 0, 0), ma=None, image=None, faceuvs=None, group=0):
        self.vertex_index = vindex
        self.ma = ma
        self.image = image
        self.faceuvs = faceuvs
        self.offset = [0, 0, 0]  # offset indices
        self.group = group


def extract_triangles(mesh):
    
    """Extract triangles from a mesh."""

    mesh.calc_loop_triangles()
    (polygroup, count) = mesh.calc_smooth_groups(use_bitflags=True)

    tri_list = []
    do_uv = bool(mesh.uv_layers)

    img = None
    for i, face in enumerate(mesh.loop_triangles):
        f_v = face.vertices

        uf = mesh.uv_layers.active.data if do_uv else None

        if do_uv:
            f_uv = [uf[lp].uv for lp in face.loops]
            for ma in mesh.materials:
                img = get_uv_image(ma) if uf else None
                if img is not None:
                    img = img.name

        smoothgroup = polygroup[face.polygon_index]

        if len(f_v) == 3:
            new_tri = tri_wrapper((f_v[0], f_v[1], f_v[2]), face.material_index, img)
            if (do_uv):
                new_tri.faceuvs = uv_key(f_uv[0]), uv_key(f_uv[1]), uv_key(f_uv[2])
            new_tri.group = smoothgroup if face.use_smooth else 0
            tri_list.append(new_tri)

    return tri_list


def remove_face_UVs(ob, mesh, bones, tri_list, vert_flags):
    
    """Remove face UV coordinates from a list of triangles.

    3DS files only supported one pair of UV coordinates per vertex, so for P2 files we convert
    Blender's face UV coordinates to vertex UV coordinates and split (i.e. duplicate) the vertices
    into new 'virtual' vertices when there are multiple UV coordinates per face/vertex."""

    verts = mesh.vertices
    
    unique_uvs = [{} for i in range(len(verts))]    # initialize a list of UniqueLists, one per vertex:

    for tri in tri_list:            # for each face UV coordinate, add it to the UniqueList of the vertex
        for i in range(3):
            
            context_uv_vert = unique_uvs[tri.vertex_index[i]]
            uvkey = tri.faceuvs[i]

            offset_index__uv_3ds = context_uv_vert.get(uvkey)

            if not offset_index__uv_3ds:
                offset_index__uv_3ds = context_uv_vert[uvkey] = len(context_uv_vert), _3ds_point_uv(uvkey)

            tri.offset[i] = offset_index__uv_3ds[0]

    # Each vertex now has a UniqueList containing any UV coordinates associated with it, but only once.
    # Next, split every vertex into 'virtual' vertices (once for each UV coordinate). And lastly,
    # update all faces (i.e. triangle vertex indices) to refer to the new virtual vertices.

    virtual_index   = 0
    index_list      = []
    vert_array      = _3ds_array()
    uv_array        = _3ds_array()
    vert_flags      = _3ds_uint_array()

    for i, blender_vertex in enumerate(verts):
        
        index_list.append(virtual_index)
        
        virtual_count = len(unique_uvs[i])                                      # Count of new, virtual vertices for the original Blender vertex
                
        for _, uv_3ds in unique_uvs[i].values():                                # Duplicate vertex for each UV associated with this vertex, if any

            vert_array.add(_3ds_point_3d(blender_vertex.co))                    # Add vertex to vertex array, including any new 'virtual' vertices
            
            uv_array.add(uv_3ds)                                                # Add UV coordinate to the UV array as is (_3ds_point_uv unneeded)
            
        for flag in calc_flags(ob, bones, blender_vertex, virtual_index, virtual_count):
            vert_flags.add(_3ds_uint(flag))                                     # Virtual vertices share Blender vertex flags (i.e. bone weights)
            
        virtual_index += len(unique_uvs[i])

    for tri in tri_list:                                                        # Update triangle vertex indices to refer to the new vertex list
        for i in range(3):
            tri.offset[i] += index_list[tri.vertex_index[i]]
        tri.vertex_index = tri.offset

    return vert_array, uv_array, tri_list, vert_flags


def calc_flags(ob, bones, blender_vertex, virtual_index, virtual_count):                # Return list of vertex flags for any/all vertices
    
    vFlags = []                                                                         # Defaults to no vertex flags available for vertex
    
    for vertexGroup in ob.vertex_groups:                                                # For multiple Vertex Groups linked to this vertex:
        try:
            weight = ob.vertex_groups[vertexGroup.index].weight(blender_vertex.index)   # If this line traps, this VGroup lacks this vertex
            weight = int(abs(weight * 100))                                             # Weight as % in bits 0-6 (scales 0.0-1.0 to 0-100)
                      
            if bones is None:                                                           # 'Action' anims need VGroups to add vertex weights
                boneIndex   = 0                                                         # (i.e. 100% weight on its vertices maps to bone[0])
            else:                                                                       # (and Action is a parentless/childless 'mono-bone')
                boneName    = ob.vertex_groups[vertexGroup.index].name                  # Use VGroup name and get the (identical!) Bone name
                boneIndex   = int(bones.keys().index(boneName) << 7)                    # Bone Index in bits 7-15 (maximum 512 bone indices)
                
            for i in range(virtual_count):                                              # For each new virtual vertex (must be at least one)
                vertexIndex = int((virtual_index + i) << 16)                            # Vertex Index in bits 16-31  (maximum 64k vertices)
                vFlags.append(vertexIndex | boneIndex | weight)                         # Encode Vertex/Bone Indices & Weight to 32-bit word
                
        except RuntimeError as exception:
            pass
        
    return vFlags


def make_faces_chunk(tri_list, mesh, materialDict):
    
    """Make a chunk for the faces.
    Also adds subchunks assigning materials to all faces."""
    
    do_smooth = False
    use_smooth = [poly.use_smooth for poly in mesh.polygons]
    if True in use_smooth:
        do_smooth = True

    materials = mesh.materials
    if not materials:
        ma = None

    face_chunk = _3ds_chunk(OBJECT_FACES)
    face_list = _3ds_array()

    if mesh.uv_layers:
        # Gather materials used in this mesh - mat/image pairs
        unique_mats = {}
        for i, tri in enumerate(tri_list):

            face_list.add(_3ds_face(tri.vertex_index))

            if materials:
                ma = materials[tri.ma]
                if ma:
                    ma = ma.name

            img = tri.image

            try:
                context_face_array = unique_mats[ma, img][1]
            except:
                name_str = ma if ma else "None"
                if img:
                    name_str += img

                context_face_array = _3ds_array()
                unique_mats[ma, img] = _3ds_string(sane_name(name_str)), context_face_array

            context_face_array.add(_3ds_ushort(i))
            # obj_material_faces[tri.ma].add(_3ds_ushort(i))

        face_chunk.add_variable("faces", face_list)
        for ma_name, ma_faces in unique_mats.values():
            obj_material_chunk = _3ds_chunk(OBJECT_MATERIAL)
            obj_material_chunk.add_variable("name", ma_name)
            obj_material_chunk.add_variable("face_list", ma_faces)
            face_chunk.add_subchunk(obj_material_chunk)

    else:

        obj_material_faces = []
        obj_material_names = []
        for m in materials:
            if m:
                obj_material_names.append(_3ds_string(sane_name(m.name)))
                obj_material_faces.append(_3ds_array())
        n_materials = len(obj_material_names)

        for i, tri in enumerate(tri_list):
            face_list.add(_3ds_face(tri.vertex_index))
            if (tri.ma < n_materials):
                obj_material_faces[tri.ma].add(_3ds_ushort(i))

        face_chunk.add_variable("faces", face_list)
        for i in range(n_materials):
            obj_material_chunk = _3ds_chunk(OBJECT_MATERIAL)
            obj_material_chunk.add_variable("name", obj_material_names[i])
            obj_material_chunk.add_variable("face_list", obj_material_faces[i])
            face_chunk.add_subchunk(obj_material_chunk)

    if do_smooth:
        obj_smooth_chunk = _3ds_chunk(OBJECT_SMOOTH)
        for i, tri in enumerate(tri_list):
            obj_smooth_chunk.add_variable("face_" + str(i), _3ds_uint(tri.group))
        face_chunk.add_subchunk(obj_smooth_chunk)

    return face_chunk


def make_vert_chunk(vert_array):
    
    """Make a vertex chunk out of an array of vertices."""
    
    vert_chunk = _3ds_chunk(OBJECT_VERTICES)
    vert_chunk.add_variable("vertices", vert_array)
    return vert_chunk


def make_vert_flag_chunk(vert_flag_array):
    
    """Make a vertex flag chunk out of an array of vertex flags."""
    
    vert_flag_chunk = _3ds_chunk(OBJECT_VERTFLAG)
    vert_flag_chunk.add_variable("vertex flags", vert_flag_array)
    return vert_flag_chunk


def make_bone_chunk(bone_array):
    
    """Make a bone chunk out of an array of struct.pack'd bone strings."""
    
    bone_chunk = _3ds_chunk(OBJECT_POSEBONES)
    bone_chunk.add_variable("armature bones", bone_array)
    return bone_chunk


def make_uv_chunk(uv_array):
    
    """Make a UV chunk out of an array of UV coordinates."""
    
    uv_chunk = _3ds_chunk(OBJECT_UV)
    uv_chunk.add_variable("uv coords", uv_array)
    return uv_chunk


def matrix_local(armature_obj, bone_name):
    
    local = armature_obj.data.bones[bone_name].matrix_local
    basis = armature_obj.pose.bones[bone_name].matrix_basis
    parent = armature_obj.pose.bones[bone_name].parent
    
    if parent == None:
        resultant = (local @ basis)
        # print("'%s' as ROOT BONE returns local @ basis:\nlocal:\n%s\nbasis:\n%s\nresultant:\n%s" % (bone_name, local, basis, resultant))
    else:
        parent_local = armature_obj.data.bones[parent.name].matrix_local
        resultant = ((parent_local.inverted() @ local) @ basis)
        # print("'%s' returns (parent_local.inverted() @local) @ basis\nparent_local.inverted():\n%s\nlocal:\n%s\nbasis:\n%s\nresultant:\n%s" % (bone_name, parent_local.inverted(), local, basis, resultant))
    
    return resultant


def make_mesh_chunk(ob, mesh, matrix, materialDict, translation):
    
    """Make a chunk out of a Blender mesh."""
    
    context = bpy.context                               # (Saves some typing)
    scene   = context.scene
    
    uv_array = None                                     # Set defaults
    armature = None
    vert_flag_array = None
    armatureBindBones = None
    armaturePoseBones = None
    output_bone_array = None
    
    tri_list = extract_triangles(mesh)                  # Extract the triangles from the mesh

    for object in scene.objects:                        # In the context of the current scene (as compared to all objects):
        if object.type != 'ARMATURE':
            continue
        else:
            armature = object
            armatureBindBones = armature.data.bones     # Get the listhead of bones' "Bind" positions; unanimated, resting-state matrices
            armaturePoseBones = armature.pose.bones     # Get the listhead of bones' "Pose" positions; *animated* transformation matrices
            break                                       # Only support *1st* armature found as bone names/indices are not unique among armatures    //TODO: Support >1 Armatures?
    
    if mesh.uv_layers:                                  # If UVs are defined, convert face UVs to vertex UVs to bypass 3DS limit of 1 UV coord/vertex
        vert_array, uv_array, tri_list, vert_flag_array = remove_face_UVs(ob, mesh, armatureBindBones, tri_list, vert_flag_array)
    else:
        vert_array = _3ds_array()                       # Otherwise, just create a new vertex array...
        
        for vert in mesh.vertices:                      # ... and add the vertices to the new array
            vert_array.add(_3ds_point_3d(vert.co))
    
    if armatureBindBones is not None:
        
        output_bone_array = _3ds_string_array()
        
        for index, bone in enumerate(armatureBindBones, start=0):                                   # For all default, resting-state Bind bones:

            boneName                    = bytes(bone.name, 'ascii')                                 # Get bone name as an array of bytes
            
            if bone.parent:                                                                         # Does this bone have a parent bone?
                parent_bone_index       = armatureBindBones.find(bone.parent.name)                  # Get parent bone index, by bone name
                localBindTransform      = bone.parent.matrix_local.inverted() @ bone.matrix_local   # Use Local-space (relative to parent)
            else:
                parent_bone_index       = 0                                                         # No parent makes this the 'root' bone
                localBindTransform      = bone.matrix_local                                         # Use Model-space (relative to origin)
         
            location                    = localBindTransform.to_translation()
            rotation                    = localBindTransform.to_quaternion()
            _,_,scale                   = localBindTransform.decompose()

            p_index                     = struct.pack("<H", index)                                  # Pack bone indices (16-bit, LSB first)
            p_parent_index              = struct.pack("<H", parent_bone_index)                      # Pack parent bone index

            if bone.children:
                p_child_indices         = struct.pack("<H", len(bone.children))                     # Pack prefix for number of children
                for childBone in bone.children:
                    p_child_indices    += struct.pack("<H", armatureBindBones.find(childBone.name)) # Pack child index as unsigned short
            else:
                p_child_indices         = struct.pack("<H", 0)                                      # Pack an empty placeholder downstream

            p_boneName                  = struct.pack("<B%ds" % (len(boneName)), len(boneName), boneName)  # Pack (variable length) bone name
            
            p_location                  = struct.pack("<3f", *mathutils.Vector(location))           # Pack items for bone animation/vertex weighting
            p_rotation                  = struct.pack("<4f", *mathutils.Quaternion(rotation))
            p_scale                     = struct.pack("<3f", *mathutils.Vector(scale))
            
            p_localBindTransform        = struct.pack("<16f", *localBindTransform[0],   *localBindTransform[1],   *localBindTransform[2],   *localBindTransform[3])
            
            packed_output_string        = p_index + p_parent_index + p_child_indices + p_boneName + p_location + p_rotation + p_scale + p_localBindTransform

            if (scene.frame_end > 0):
                p_anim_transforms       = struct.pack("<H", scene.frame_end)                        # unsigned short (i.e. 64K) frames really should be enough
            else:
                p_anim_transforms       = struct.pack("<H", 0)
                
            packed_output_string       += p_anim_transforms                                         # Append prefix of this bone's animation transform array
                        
            for f in range(scene.frame_start, scene.frame_end+1):                                   # Every bone has all frames, even if keyframeless
                scene.frame_set(f)                                                                  # Position the keyframe to the current index
                if armaturePoseBones is not None:
                    for pose_bone in armaturePoseBones:                                             # For every animated pose bone available:
                        if (pose_bone.name == bone.name):                                           # Find the current bone name
                            boneAnimation   = matrix_local(armature, pose_bone.name)                # Get the animation matrix
                            p_boneAnimation = struct.pack("<16f", *boneAnimation[0],  *boneAnimation[1],  *boneAnimation[2],  *boneAnimation[3])
                            
                            packed_output_string += p_boneAnimation                                 # Append animation matrix to output

            output_bone_array.add(_3ds_packed_string(packed_output_string))                         # Write out packed output string

    # create the mesh chunk:
    
    mesh_chunk = _3ds_chunk(OBJECT_MESH)

    # add vertex chunk:
    
    mesh_chunk.add_subchunk(make_vert_chunk(vert_array))

    # if available, add vertex flag chunk:
    
    if vert_flag_array:
        mesh_chunk.add_subchunk(make_vert_flag_chunk(vert_flag_array))
    
    #if available, add a bones chunk:
    
    if output_bone_array:
        mesh_chunk.add_subchunk(make_bone_chunk(output_bone_array))
    
    # add faces chunk:
    
    mesh_chunk.add_subchunk(make_faces_chunk(tri_list, mesh, materialDict))

    # if available, add uv chunk:
    
    if uv_array:
        mesh_chunk.add_subchunk(make_uv_chunk(uv_array))

    # create transformation matrix chunk:
    
    matrix_chunk = _3ds_chunk(OBJECT_MESH_MATRIX)
    
    if ob.parent is None:
        obj_translate = translation[ob.name]

    else:  # Calculate child matrix translation relative to parent:
        obj_translate = (ob.parent.matrix_local.inverted() @ ob.matrix_local).to_translation()

    matrix_chunk.add_variable("xx", _3ds_float(matrix[0].to_tuple(6)[0]))
    matrix_chunk.add_variable("xy", _3ds_float(matrix[0].to_tuple(6)[1]))
    matrix_chunk.add_variable("xz", _3ds_float(matrix[0].to_tuple(6)[2]))
    matrix_chunk.add_variable("xw", _3ds_float(0))
    matrix_chunk.add_variable("yx", _3ds_float(matrix[1].to_tuple(6)[0]))
    matrix_chunk.add_variable("yy", _3ds_float(matrix[1].to_tuple(6)[1]))
    matrix_chunk.add_variable("yz", _3ds_float(matrix[1].to_tuple(6)[2]))
    matrix_chunk.add_variable("yw", _3ds_float(0))
    matrix_chunk.add_variable("zx", _3ds_float(matrix[2].to_tuple(6)[0]))
    matrix_chunk.add_variable("zy", _3ds_float(matrix[2].to_tuple(6)[1]))
    matrix_chunk.add_variable("zz", _3ds_float(matrix[2].to_tuple(6)[2]))
    matrix_chunk.add_variable("zw", _3ds_float(0))
    matrix_chunk.add_variable("tx", _3ds_float(obj_translate.to_tuple(6)[0]))
    matrix_chunk.add_variable("ty", _3ds_float(obj_translate.to_tuple(6)[1]))
    matrix_chunk.add_variable("tz", _3ds_float(obj_translate.to_tuple(6)[2])) 
    matrix_chunk.add_variable("tw", _3ds_float(1)) 

    mesh_chunk.add_subchunk(matrix_chunk)

    return mesh_chunk


def calc_target(posi, tilt=0.0, pan=0.0):
    
    """Calculate target position for cameras and spotlights."""
    
    adjacent = math.radians(90)
    turn = 0.0 if abs(pan) < adjacent else -0.0
    lean = 0.0 if abs(tilt) > adjacent else -0.0
    diagonal = math.sqrt(pow(posi.x ,2) + pow(posi.y ,2))
    target_x = math.copysign(posi.x + (posi.y * math.tan(pan)), pan)
    target_y = math.copysign(posi.y + (posi.x * math.tan(adjacent - pan)), turn)
    target_z = math.copysign(posi.z + diagonal * math.tan(adjacent - tilt), lean)

    return target_x, target_y, target_z


def make_kfdata(start=0, stop=100, curtime=0):
    
    """Make the basic keyframe data chunk"""
    
    kfdata = _3ds_chunk(KFDATA)

    kfhdr = _3ds_chunk(KFDATA_KFHDR)
    kfhdr.add_variable("revision", _3ds_ushort(0x0006))
    kfhdr.add_variable("filename", _3ds_string(sane_name("blender_io_scene_p2")))
    kfhdr.add_variable("animlen", _3ds_ushort(stop-start))

    kfseg = _3ds_chunk(KFDATA_KFSEG)
    kfseg.add_variable("start", _3ds_ushort(start))
    kfseg.add_variable("stop", _3ds_ushort(stop))

    kfcurtime = _3ds_chunk(KFDATA_KFCURTIME)
    kfcurtime.add_variable("curtime", _3ds_ushort(curtime))

    kfdata.add_subchunk(kfhdr)
    kfdata.add_subchunk(kfseg)
    kfdata.add_subchunk(kfcurtime)
    
    return kfdata


def make_track_chunk(ID, ob, ob_pos, ob_rot, ob_size):
    
    """Make a chunk for track data. Depending on the ID, this will construct
    a position, rotation, scale, roll, color, fov, hotspot or falloff track."""
    
    track_chunk = _3ds_chunk(ID)

    if ID in {POS_TRACK_TAG, ROT_TRACK_TAG, SCL_TRACK_TAG, ROLL_TRACK_TAG} and ob.animation_data and ob.animation_data.action:
        action = ob.animation_data.action
        if action.fcurves:
            fcurves = action.fcurves
            fcurves.update()
            kframes = [kf.co[0] for kf in [fc for fc in fcurves if fc is not None][0].keyframe_points]
            nkeys = len(kframes)
            if not 0 in kframes:
                kframes.append(0)
                nkeys += 1
            kframes = sorted(set(kframes))
            track_chunk.add_variable("track_flags", _3ds_ushort(0x40))
            track_chunk.add_variable("frame_start", _3ds_uint(int(action.frame_start)))
            track_chunk.add_variable("frame_end", _3ds_uint(int(action.frame_range[1])))
            track_chunk.add_variable("nkeys", _3ds_uint(nkeys))

            if ID == POS_TRACK_TAG:  # Position
                for i, frame in enumerate(kframes):
                    pos_track = [fc for fc in fcurves if fc is not None and fc.data_path == 'location']
                    pos_x = next((tc.evaluate(frame) for tc in pos_track if tc.array_index == 0), ob_pos.x)
                    pos_y = next((tc.evaluate(frame) for tc in pos_track if tc.array_index == 1), ob_pos.y)
                    pos_z = next((tc.evaluate(frame) for tc in pos_track if tc.array_index == 2), ob_pos.z)
                    pos = ob_size @ mathutils.Vector((pos_x, pos_y, pos_z))
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("position", _3ds_point_3d((pos.x, pos.y, pos.z)))

            elif ID == ROT_TRACK_TAG:  # Rotation
                for i, frame in enumerate(kframes):
                    rot_track = [fc for fc in fcurves if fc is not None and fc.data_path == 'rotation_euler']
                    rot_x = next((tc.evaluate(frame) for tc in rot_track if tc.array_index == 0), ob_rot.x)
                    rot_y = next((tc.evaluate(frame) for tc in rot_track if tc.array_index == 1), ob_rot.y)
                    rot_z = next((tc.evaluate(frame) for tc in rot_track if tc.array_index == 2), ob_rot.z)
                    quat = mathutils.Euler((rot_x, rot_y, rot_z)).to_quaternion().inverted()
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("rotation", _3ds_point_4d((quat.angle, quat.axis.x, quat.axis.y, quat.axis.z)))

            elif ID == SCL_TRACK_TAG:  # Scale
                for i, frame in enumerate(kframes):
                    scale_track = [fc for fc in fcurves if fc is not None and fc.data_path == 'scale']
                    size_x = next((tc.evaluate(frame) for tc in scale_track if tc.array_index == 0), ob_size.x)
                    size_y = next((tc.evaluate(frame) for tc in scale_track if tc.array_index == 1), ob_size.y)
                    size_z = next((tc.evaluate(frame) for tc in scale_track if tc.array_index == 2), ob_size.z)
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("scale", _3ds_point_3d((size_x, size_y, size_z)))

            elif ID == ROLL_TRACK_TAG:  # Roll
                for i, frame in enumerate(kframes):
                    roll_track = [fc for fc in fcurves if fc is not None and fc.data_path == 'rotation_euler']
                    roll = next((tc.evaluate(frame) for tc in roll_track if tc.array_index == 1), ob_rot.y)
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("roll", _3ds_float(round(math.degrees(roll), 4)))

    elif ID in {COL_TRACK_TAG, FOV_TRACK_TAG, HOTSPOT_TRACK_TAG, FALLOFF_TRACK_TAG} and ob.data.animation_data and ob.data.animation_data.action:
        action = ob.data.animation_data.action
        if action.fcurves:
            fcurves = action.fcurves
            fcurves.update()
            kframes = [kf.co[0] for kf in [fc for fc in fcurves if fc is not None][0].keyframe_points]
            nkeys = len(kframes)
            if not 0 in kframes:
                kframes.append(0)
                nkeys += 1
            kframes = sorted(set(kframes))
            track_chunk.add_variable("track_flags", _3ds_ushort(0x40))
            track_chunk.add_variable("frame_start", _3ds_uint(int(action.frame_start)))
            track_chunk.add_variable("frame_total", _3ds_uint(int(action.frame_end)))
            track_chunk.add_variable("nkeys", _3ds_uint(nkeys))

            if ID == COL_TRACK_TAG:  # Color
                for i, frame in enumerate(kframes):
                    color = [fc.evaluate(frame) for fc in fcurves if fc is not None and fc.data_path == 'color']
                    if not color:
                        color = ob.data.color[:3]
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("color", _3ds_float_color(color))

            elif ID == FOV_TRACK_TAG:  # Field of view
                for i, frame in enumerate(kframes):
                    lens = next((fc.evaluate(frame) for fc in fcurves if fc is not None and fc.data_path == 'lens'), ob.data.lens)
                    fov = 2 * math.atan(ob.data.sensor_width / (2 * lens))
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("fov", _3ds_float(round(math.degrees(fov), 4)))

            elif ID == HOTSPOT_TRACK_TAG:  # Hotspot
                for i, frame in enumerate(kframes):
                    beamsize = next((fc.evaluate(frame) for fc in fcurves if fc is not None and fc.data_path == 'spot_size'), ob.data.spot_size)
                    blend = next((fc.evaluate(frame) for fc in fcurves if fc is not None and fc.data_path == 'spot_blend'), ob.data.spot_blend)
                    hot_spot = math.degrees(beamsize) - (blend * math.floor(math.degrees(beamsize)))
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("hotspot", _3ds_float(round(hot_spot, 4)))

            elif ID == FALLOFF_TRACK_TAG:  # Falloff
                for i, frame in enumerate(kframes):
                    fall_off = next((fc.evaluate(frame) for fc in fcurves if fc is not None and fc.data_path == 'spot_size'), ob.data.spot_size)
                    track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                    track_chunk.add_variable("tcb_flags", _3ds_ushort())
                    track_chunk.add_variable("falloff", _3ds_float(round(math.degrees(fall_off), 4)))

    else:
        track_chunk.add_variable("track_flags", _3ds_ushort(0x40))  # Based on observation default flag is 0x40
        track_chunk.add_variable("frame_start", _3ds_uint(0))
        track_chunk.add_variable("frame_total", _3ds_uint(0))
        track_chunk.add_variable("nkeys", _3ds_uint(1))
        # Next section should be repeated for every keyframe, with no animation only one tag is needed
        track_chunk.add_variable("tcb_frame", _3ds_uint(0))
        track_chunk.add_variable("tcb_flags", _3ds_ushort())

        # New method simply inserts the parameters
        if ID == POS_TRACK_TAG:  # Position vector
            track_chunk.add_variable("position", _3ds_point_3d(ob_pos))

        elif ID == ROT_TRACK_TAG:  # Rotation (angle first [radians], followed by axis)
            quat = ob_rot.to_quaternion().inverted()
            track_chunk.add_variable("rotation", _3ds_point_4d((quat.angle, quat.axis.x, quat.axis.y, quat.axis.z)))

        elif ID == SCL_TRACK_TAG:  # Scale vector
            track_chunk.add_variable("scale", _3ds_point_3d(ob_size))

        elif ID == ROLL_TRACK_TAG:  # Roll angle
            track_chunk.add_variable("roll", _3ds_float(round(math.degrees(ob_rot.y), 4)))

        elif ID == COL_TRACK_TAG:  # Color values
            track_chunk.add_variable("color", _3ds_float_color(ob.data.color[:3]))

        elif ID == FOV_TRACK_TAG:  # Field of view
            track_chunk.add_variable("fov", _3ds_float(round(math.degrees(ob.data.angle), 4)))

        elif ID == HOTSPOT_TRACK_TAG:  # Hotspot
            beam_angle = math.degrees(ob.data.spot_size)
            track_chunk.add_variable("hotspot", _3ds_float(round(beam_angle - (ob.data.spot_blend * math.floor(beam_angle)), 4)))

        elif ID == FALLOFF_TRACK_TAG:  # Falloff
            track_chunk.add_variable("falloff", _3ds_float(round(math.degrees(ob.data.spot_size), 4)))

    return track_chunk


def make_object_node(ob, translation, rotation, scale, name_id):
    
    """Make a node chunk for a Blender object. Takes Blender object as parameter.
       Blender Empty objects are converted to dummy nodes."""

    name = ob.name
    if ob.type == 'CAMERA':
        obj_node = _3ds_chunk(CAMERA_NODE_TAG)
    elif ob.type == 'LIGHT':
        obj_node = _3ds_chunk(LIGHT_NODE_TAG)
        if ob.data.type == 'SPOT':
            obj_node = _3ds_chunk(SPOT_NODE_TAG)
    else:  # Main object node chunk
        obj_node = _3ds_chunk(OBJECT_NODE_TAG)

    # Chunk for the object ID from name_id dictionary:
    obj_id_chunk = _3ds_chunk(OBJECT_NODE_ID)
    obj_id_chunk.add_variable("node_id", _3ds_ushort(name_id[name]))    # IOW: node_id is the mesh's index number in Blender's collection of mesh objects
    obj_node.add_subchunk(obj_id_chunk)

    # Object node header with object name
    obj_node_header_chunk = _3ds_chunk(OBJECT_NODE_HDR)

    if ob.type == 'EMPTY':  # Forcing to use the real name for empties
        # Empties are called $$$DUMMY and use OBJECT_INSTANCENAME chunk as name (see below)
        # flags1 is usually 0x4000 for empty objects, otherwise 0x0040
        # flags2 is usually 0x0000, or use 0x01 for display path, 0x04 object frozen, 0x10 for motion blur, 0x20 for material morph and bit 0x40 for mesh morph
        obj_node_header_chunk.add_variable("name", _3ds_string(b"$$$DUMMY"))
        obj_node_header_chunk.add_variable("flags1", _3ds_ushort(0x4000))
        obj_node_header_chunk.add_variable("flags2", _3ds_ushort(0))

    else:
        obj_node_header_chunk.add_variable("name", _3ds_string(sane_name(name)))
        obj_node_header_chunk.add_variable("flags1", _3ds_ushort(0x0040))
        obj_node_header_chunk.add_variable("flags2", _3ds_ushort(0))

    parent = ob.parent

    if parent is None or parent.name not in name_id:
        # If no parent, or parents name is not in dictionary, ID becomes -1:
        obj_node_header_chunk.add_variable("parent", _3ds_ushort(ROOT_OBJECT))
    else:  # Get the parent's ID from the name_id dictionary:
        obj_node_header_chunk.add_variable("parent", _3ds_ushort(name_id[parent.name]))

    # Add subchunk for node header
    obj_node.add_subchunk(obj_node_header_chunk)

    # Alternatively, use PARENT_NAME chunk for hierachy
    if parent is not None and (parent.name in name_id):
        obj_parent_name_chunk = _3ds_chunk(OBJECT_PARENT_NAME)
        obj_parent_name_chunk.add_variable("parent", _3ds_string(sane_name(parent.name)))
        obj_node.add_subchunk(obj_parent_name_chunk)

    # Empty objects need to have an extra chunk for the instance name
    if ob.type == 'EMPTY':  # Will use a real object name for empties for now
        obj_instance_name_chunk = _3ds_chunk(OBJECT_INSTANCENAME)
        obj_instance_name_chunk.add_variable("name", _3ds_string(sane_name(name)))
        obj_node.add_subchunk(obj_instance_name_chunk)

    if ob.type in {'MESH', 'EMPTY'}:  # Add a pivot point at the object center
        pivot_pos = (translation[name])
        obj_pivot_chunk = _3ds_chunk(OBJECT_PIVOT)
        obj_pivot_chunk.add_variable("pivot", _3ds_point_3d(pivot_pos))
        obj_node.add_subchunk(obj_pivot_chunk)

        # Create a bounding box from quadrant diagonal
        obj_boundbox = _3ds_chunk(OBJECT_BOUNDBOX)
        obj_boundbox.add_variable("min", _3ds_point_3d(ob.bound_box[0]))
        obj_boundbox.add_variable("max", _3ds_point_3d(ob.bound_box[6]))
        obj_node.add_subchunk(obj_boundbox)

    # Add track chunks for position, rotation, size
    ob_scale = scale[name]  # and collect masterscale
    if parent is None or (parent.name not in name_id):
        ob_pos = translation[name]
        ob_rot = rotation[name]
        ob_size = ob.scale

    else:  # Calculate child position and rotation of the object center, no scale applied
        ob_pos = translation[name] - translation[parent.name]
        ob_rot = rotation[name].to_quaternion().cross(rotation[parent.name].to_quaternion().copy().inverted()).to_euler()
        ob_size = mathutils.Vector((1.0, 1.0, 1.0))

    obj_node.add_subchunk(make_track_chunk(POS_TRACK_TAG, ob, ob_pos, ob_rot, ob_scale))

    if ob.type in {'MESH', 'EMPTY'}:
        obj_node.add_subchunk(make_track_chunk(ROT_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))
        obj_node.add_subchunk(make_track_chunk(SCL_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))
    if ob.type =='CAMERA':
        obj_node.add_subchunk(make_track_chunk(FOV_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))
        obj_node.add_subchunk(make_track_chunk(ROLL_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))
    if ob.type =='LIGHT':
        obj_node.add_subchunk(make_track_chunk(COL_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))
    if ob.type == 'LIGHT' and ob.data.type == 'SPOT':
        obj_node.add_subchunk(make_track_chunk(HOTSPOT_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))
        obj_node.add_subchunk(make_track_chunk(FALLOFF_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))
        obj_node.add_subchunk(make_track_chunk(ROLL_TRACK_TAG, ob, ob_pos, ob_rot, ob_size))

    return obj_node


def make_target_node(ob, translation, rotation, scale, name_id):
    
    """Make a target chunk for light and camera objects."""

    name = ob.name
    name_id["ø " + name] = len(name_id)
    if ob.type == 'CAMERA':  # Add camera target
        tar_node = _3ds_chunk(TARGET_NODE_TAG)
    elif ob.type == 'LIGHT':  # Add spot target
        tar_node = _3ds_chunk(LTARGET_NODE_TAG)

    # Chunk for the object ID from name_id dictionary:
    tar_id_chunk = _3ds_chunk(OBJECT_NODE_ID)
    tar_id_chunk.add_variable("node_id", _3ds_ushort(name_id[name]))
    tar_node.add_subchunk(tar_id_chunk)

    # Object node header with object name
    tar_node_header_chunk = _3ds_chunk(OBJECT_NODE_HDR)
    # Targets get the same name as the object, flags1 is usually 0x0010 and parent set to ROOT_OBJECT
    tar_node_header_chunk.add_variable("name", _3ds_string(sane_name(name)))
    tar_node_header_chunk.add_variable("flags1", _3ds_ushort(0x0010))
    tar_node_header_chunk.add_variable("flags2", _3ds_ushort(0))
    tar_node_header_chunk.add_variable("parent", _3ds_ushort(ROOT_OBJECT))

    # Add subchunk for node header
    tar_node.add_subchunk(tar_node_header_chunk)

    # Calculate target position
    ob_pos = translation[name]
    ob_rot = rotation[name]
    ob_scale = scale[name]
    target_pos = calc_target(ob_pos, ob_rot.x, ob_rot.z)

    # Add track chunks for target position
    track_chunk = _3ds_chunk(POS_TRACK_TAG)

    if ob.animation_data and ob.animation_data.action:
        action = ob.animation_data.action
        if action.fcurves:
            fcurves = action.fcurves
            fcurves.update()
            kframes = [kf.co[0] for kf in [fc for fc in fcurves if fc is not None][0].keyframe_points]
            nkeys = len(kframes)
            if not 0 in kframes:
                kframes.append(0)
                nkeys += 1
            kframes = sorted(set(kframes))
            track_chunk.add_variable("track_flags", _3ds_ushort(0x40))
            track_chunk.add_variable("frame_start", _3ds_uint(int(action.frame_start)))
            track_chunk.add_variable("frame_end", _3ds_uint(int(action.frame_range[1])))
            track_chunk.add_variable("nkeys", _3ds_uint(nkeys))

            for i, frame in enumerate(kframes):
                loc_target = [fc for fc in fcurves if fc is not None and fc.data_path == 'location']
                loc_x = next((tc.evaluate(frame) for tc in loc_target if tc.array_index == 0), ob_pos.x)
                loc_y = next((tc.evaluate(frame) for tc in loc_target if tc.array_index == 1), ob_pos.y)
                loc_z = next((tc.evaluate(frame) for tc in loc_target if tc.array_index == 2), ob_pos.z)
                rot_target = [fc for fc in fcurves if fc is not None and fc.data_path == 'rotation_euler']
                rot_x = next((tc.evaluate(frame) for tc in rot_target if tc.array_index == 0), ob_rot.x)
                rot_z = next((tc.evaluate(frame) for tc in rot_target if tc.array_index == 2), ob_rot.z)
                target_distance = ob_scale @ mathutils.Vector((loc_x, loc_y, loc_z))
                target_pos = calc_target(target_distance, rot_x, rot_z)
                track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                track_chunk.add_variable("tcb_flags", _3ds_ushort())
                track_chunk.add_variable("position", _3ds_point_3d(target_pos))

    else:  # Track header
        track_chunk.add_variable("track_flags", _3ds_ushort(0x40))  # Based on observation, default flag is 0x40
        track_chunk.add_variable("frame_start", _3ds_uint(0))
        track_chunk.add_variable("frame_total", _3ds_uint(0))
        track_chunk.add_variable("nkeys", _3ds_uint(1))
        # Keyframe header
        track_chunk.add_variable("tcb_frame", _3ds_uint(0))
        track_chunk.add_variable("tcb_flags", _3ds_ushort())
        track_chunk.add_variable("position", _3ds_point_3d(target_pos))

    tar_node.add_subchunk(track_chunk)

    return tar_node

def make_ambient_node(world):
    
    """Make an ambient node for the world color, if the color is animated."""

    amb_color = world.color[:3]
    amb_node = _3ds_chunk(AMBIENT_NODE_TAG)
    track_chunk = _3ds_chunk(COL_TRACK_TAG)

    # Chunk for the ambient ID is ROOT_OBJECT
    amb_id_chunk = _3ds_chunk(OBJECT_NODE_ID)
    amb_id_chunk.add_variable("node_id", _3ds_ushort(ROOT_OBJECT))
    amb_node.add_subchunk(amb_id_chunk)

    # Object node header, name is "$AMBIENT$" for ambient nodes
    amb_node_header_chunk = _3ds_chunk(OBJECT_NODE_HDR)
    amb_node_header_chunk.add_variable("name", _3ds_string(b"$AMBIENT$"))
    amb_node_header_chunk.add_variable("flags1", _3ds_ushort(0x4000))  # Flags1 0x4000 for empty objects
    amb_node_header_chunk.add_variable("flags2", _3ds_ushort(0))
    amb_node_header_chunk.add_variable("parent", _3ds_ushort(ROOT_OBJECT))
    amb_node.add_subchunk(amb_node_header_chunk)

    if world.use_nodes and world.node_tree.animation_data.action:
        ambioutput = 'EMISSION' ,'MIX_SHADER', 'WORLD_OUTPUT'
        action = world.node_tree.animation_data.action
        links = world.node_tree.links
        ambilinks = [lk for lk in links if lk.from_node.type in {'EMISSION', 'RGB'} and lk.to_node.type in ambioutput]
        if ambilinks and action.fcurves:
            fcurves = action.fcurves
            fcurves.update()
            emission = next((lk.from_socket.node for lk in ambilinks if lk.to_node.type in ambioutput), False)
            ambinode = next((lk.from_socket.node for lk in ambilinks if lk.to_node.type == 'EMISSION'), emission)
            kframes = [kf.co[0] for kf in [fc for fc in fcurves if fc is not None][0].keyframe_points]
            ambipath = ('nodes[\"RGB\"].outputs[0].default_value' if ambinode and ambinode.type == 'RGB' else
                        'nodes[\"Emission\"].inputs[0].default_value')
            nkeys = len(kframes)
            if not 0 in kframes:
                kframes.append(0)
                nkeys = nkeys + 1
            kframes = sorted(set(kframes))
            track_chunk.add_variable("track_flags", _3ds_ushort(0x40))
            track_chunk.add_variable("frame_start", _3ds_uint(int(action.frame_start)))
            track_chunk.add_variable("frame_total", _3ds_uint(int(action.frame_end)))
            track_chunk.add_variable("nkeys", _3ds_uint(nkeys))

            for i, frame in enumerate(kframes):
                ambient = [fc.evaluate(frame) for fc in fcurves if fc is not None and fc.data_path == ambipath]
                if not ambient:
                    ambient = amb_color
                track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                track_chunk.add_variable("tcb_flags", _3ds_ushort())
                track_chunk.add_variable("color", _3ds_float_color(ambient[:3]))

    elif world.animation_data.action:
        action = world.animation_data.action
        if action.fcurves:
            fcurves = action.fcurves
            fcurves.update()
            kframes = [kf.co[0] for kf in [fc for fc in fcurves if fc is not None][0].keyframe_points]
            nkeys = len(kframes)
            if not 0 in kframes:
                kframes.append(0)
                nkeys += 1
            kframes = sorted(set(kframes))
            track_chunk.add_variable("track_flags", _3ds_ushort(0x40))
            track_chunk.add_variable("frame_start", _3ds_uint(int(action.frame_start)))
            track_chunk.add_variable("frame_total", _3ds_uint(int(action.frame_end)))
            track_chunk.add_variable("nkeys", _3ds_uint(nkeys))

            for i, frame in enumerate(kframes):
                ambient = [fc.evaluate(frame) for fc in fcurves if fc is not None and fc.data_path == 'color']
                if not ambient:
                    ambient = amb_color
                track_chunk.add_variable("tcb_frame", _3ds_uint(int(frame)))
                track_chunk.add_variable("tcb_flags", _3ds_ushort())
                track_chunk.add_variable("color", _3ds_float_color(ambient))

    else:  # Track header
        track_chunk.add_variable("track_flags", _3ds_ushort(0x40))
        track_chunk.add_variable("frame_start", _3ds_uint(0))
        track_chunk.add_variable("frame_total", _3ds_uint(0))
        track_chunk.add_variable("nkeys", _3ds_uint(1))
        # Keyframe header
        track_chunk.add_variable("tcb_frame", _3ds_uint(0))
        track_chunk.add_variable("tcb_flags", _3ds_ushort())
        track_chunk.add_variable("color", _3ds_float_color(amb_color))

    amb_node.add_subchunk(track_chunk)

    return amb_node


def save(operator, context, filepath="", scale_factor=1.0, use_scene_unit=False, use_selection=False,
         object_filter=None, use_hierarchy=False, use_keyframes=True, global_matrix=None, use_cursor=False):
    
    """Save the Blender scene to a .p2 file."""

    blender_version = getattr(bpy.app, "version", (0, 0, 0))
    if blender_version >= (5, 0, 0):
        message = ("P-Squared exporter currently supports Blender 4.5 or earlier. "
                   f"Detected Blender {blender_version[0]}.{blender_version[1]}. "
                   "Use Blender 4.5 for reliable exports until 5.x support arrives.")
        if operator and hasattr(operator, "report"):
            operator.report({'ERROR'}, message)
        else:
            print(message)
        raise RuntimeError(message)

    import time
    
    duration = time.time()
    context.window.cursor_set('WAIT')

    scene = context.scene
    layer = context.view_layer
    depsgraph = context.evaluated_depsgraph_get()
    world = scene.world

    unit_measure = 1.0
    if use_scene_unit:
        unit_length = scene.unit_settings.length_unit
        if unit_length == 'MILES':
            unit_measure = 0.000621371
        elif unit_length == 'KILOMETERS':
            unit_measure = 0.001
        elif unit_length == 'FEET':
            unit_measure = 3.280839895
        elif unit_length == 'INCHES':
            unit_measure = 39.37007874
        elif unit_length == 'CENTIMETERS':
            unit_measure = 100
        elif unit_length == 'MILLIMETERS':
            unit_measure = 1000
        elif unit_length == 'THOU':
            unit_measure = 39370.07874
        elif unit_length == 'MICROMETERS':
            unit_measure = 1000000

    mtx_scale = mathutils.Matrix.Scale((scale_factor * unit_measure),4)

    if global_matrix is None:
        global_matrix = mathutils.Matrix()

    if bpy.ops.object.mode_set.poll():
        bpy.ops.object.mode_set(mode='OBJECT')

    # Initialize the primary chunk
    primary = _3ds_chunk(M3DMAGIC)

    # Add version chunk
    version_chunk = _3ds_chunk(M3D_VERSION)
    version_chunk.add_variable("version", _3ds_uint(3))
    primary.add_subchunk(version_chunk)

    # Init main object info chunk
    object_info = _3ds_chunk(MDATA)
    mesh_version = _3ds_chunk(MESH_VERSION)
    mesh_version.add_variable("mesh", _3ds_uint(3))
    object_info.add_subchunk(mesh_version)

    # Init main keyframe data chunk
    if use_keyframes:
        stop = scene.frame_end
        start = scene.frame_start
        curtime = scene.frame_current
        kfdata = make_kfdata(start, stop, curtime)

    # Make a list of all materials used in the selected meshes (use dictionary, so each material is added only once)
    materialDict = {}
    mesh_objects = []

    if use_selection:
        objects = [ob for ob in scene.objects if ob.type in object_filter and ob.visible_get(view_layer=layer) and ob.select_get(view_layer=layer)]
    else:
        objects = [ob for ob in scene.objects if ob.type in object_filter and ob.visible_get(view_layer=layer)]

    empty_objects = [ob for ob in objects if ob.type == 'EMPTY']
    light_objects = [ob for ob in objects if ob.type == 'LIGHT']
    camera_objects = [ob for ob in objects if ob.type == 'CAMERA']

    for ob in objects:
        # Get derived objects
        derived_dict = bpy_extras.io_utils.create_derived_objects(depsgraph, [ob])
        derived = derived_dict.get(ob)

        if derived is None:
            continue

        for ob_derived, mtx in derived:
            if ob.type not in {'MESH', 'CURVE', 'SURFACE', 'FONT', 'META'}:
                continue

            try:
                data = ob_derived.to_mesh()
            except:
                data = None

            if data:
                matrix = global_matrix @ mtx
                data.transform(matrix)
                data.transform(mtx_scale)
                mesh_objects.append((ob_derived, data, matrix))
                ma_ls = data.materials
                ma_ls_len = len(ma_ls)

                # Get material/image tuples
                if data.uv_layers:
                    if not ma_ls:
                        ma = ma_name = None

                    for f, uf in zip(data.polygons, data.uv_layers.active.data):
                        if ma_ls:
                            ma_index = f.material_index
                            if ma_index >= ma_ls_len:
                                ma_index = f.material_index = 0
                            ma = ma_ls[ma_index]
                            ma_name = None if ma is None else ma.name
                        # Else there already set to none

                        img = get_uv_image(ma)
                        img_name = None if img is None else img.name

                        materialDict.setdefault((ma_name, img_name), (ma, img))

                else:
                    for ma in ma_ls:
                        if ma:  # Material may be None so check its not
                            materialDict.setdefault((ma.name, None), (ma, None))

                    # Why 0 Why!
                    for f in data.polygons:
                        if f.material_index >= ma_ls_len:
                            f.material_index = 0


    # Make MATERIAL chunks for all materials used in the meshes
    for ma_image in materialDict.values():
        object_info.add_subchunk(make_material_chunk(ma_image[0], ma_image[1]))

    # Add MASTERSCALE element, if any. May not have one if no Armature object available
    if bpy.context.active_object is None:
        scale_vector = struct.pack("<3f", *mathutils.Vector((1.0, 1.0, 1.0)))
    else:
        scale_vector = struct.pack("<3f", *mathutils.Vector(bpy.context.active_object.scale))  # Pack scale vector

    mscale = _3ds_chunk(MASTERSCALE)
    mscale.add_variable("scale", _3ds_packed_string(scale_vector))
    object_info.add_subchunk(mscale)

    # Add 3D cursor location
    if use_cursor:
        cursor_chunk = _3ds_chunk(O_CONSTS)
        cursor_chunk.add_variable("cursor", _3ds_point_3d(scene.cursor.location))
        object_info.add_subchunk(cursor_chunk)

    # Add AMBIENT color
    if world is not None and 'WORLD' in object_filter:
        ambient_chunk = _3ds_chunk(AMBIENT_LIGHT)
        ambient_light = _3ds_chunk(RGB)
        ambient_light.add_variable("ambient", _3ds_float_color(world.color))
        ambient_chunk.add_subchunk(ambient_light)
        object_info.add_subchunk(ambient_chunk)

        # Add BACKGROUND and BITMAP
        if world.use_nodes:
            bgtype = 'BACKGROUND'
            ntree = world.node_tree.links
            background_color_chunk = _3ds_chunk(RGB)
            background_chunk = _3ds_chunk(SOLIDBACKGND)
            background_flag = _3ds_chunk(USE_SOLIDBGND)
            bgmixer = 'BACKGROUND', 'MIX', 'MIX_RGB'
            bgshade = 'ADD_SHADER', 'MIX_SHADER', 'OUTPUT_WORLD'
            bg_tex = 'TEX_IMAGE', 'TEX_ENVIRONMENT'
            bg_color = next((lk.from_node.inputs[0].default_value[:3] for lk in ntree if lk.from_node.type == bgtype and lk.to_node.type in bgshade), world.color)
            bg_mixer = next((lk.from_node.type for lk in ntree if  lk.from_node.type in bgmixer and lk.to_node.type == bgtype), bgtype)
            bg_image = next((lk.from_node.image for lk in ntree if lk.from_node.type in bg_tex and lk.to_node.type == bg_mixer), False)
            gradient = next((lk.from_node.color_ramp.elements for lk in ntree if lk.from_node.type == 'VALTORGB' and lk.to_node.type in bgmixer), False)
            background_color_chunk.add_variable("color", _3ds_float_color(bg_color))
            background_chunk.add_subchunk(background_color_chunk)
            if bg_image and bg_image is not None:
                background_image = _3ds_chunk(BITMAP)
                background_flag = _3ds_chunk(USE_BITMAP)
                background_image.add_variable("image", _3ds_string(sane_name(bg_image.name)))
                object_info.add_subchunk(background_image)
            object_info.add_subchunk(background_chunk)

            # Add VGRADIENT chunk
            if gradient and len(gradient) >= 3:
                gradient_chunk = _3ds_chunk(VGRADIENT)
                background_flag = _3ds_chunk(USE_VGRADIENT)
                gradient_chunk.add_variable("midpoint", _3ds_float(gradient[1].position))
                gradient_topcolor_chunk = _3ds_chunk(RGB)
                gradient_topcolor_chunk.add_variable("color", _3ds_float_color(gradient[2].color[:3]))
                gradient_chunk.add_subchunk(gradient_topcolor_chunk)
                gradient_midcolor_chunk = _3ds_chunk(RGB)
                gradient_midcolor_chunk.add_variable("color", _3ds_float_color(gradient[1].color[:3]))
                gradient_chunk.add_subchunk(gradient_midcolor_chunk)
                gradient_lowcolor_chunk = _3ds_chunk(RGB)
                gradient_lowcolor_chunk.add_variable("color", _3ds_float_color(gradient[0].color[:3]))
                gradient_chunk.add_subchunk(gradient_lowcolor_chunk)
                object_info.add_subchunk(gradient_chunk)
            object_info.add_subchunk(background_flag)

            # Add FOG
            fognode = next((lk.from_socket.node for lk in ntree if lk.from_socket.node.type == 'VOLUME_ABSORPTION' and lk.to_socket.node.type in bgshade), False)
            if fognode:
                fog_chunk = _3ds_chunk(FOG)
                fog_color_chunk = _3ds_chunk(RGB)
                use_fog_flag = _3ds_chunk(USE_FOG)
                fog_density = fognode.inputs['Density'].default_value * 100
                fog_color_chunk.add_variable("color", _3ds_float_color(fognode.inputs[0].default_value[:3]))
                fog_chunk.add_variable("nearplane", _3ds_float(world.mist_settings.start))
                fog_chunk.add_variable("nearfog", _3ds_float(fog_density * 0.5))
                fog_chunk.add_variable("farplane", _3ds_float(world.mist_settings.depth))
                fog_chunk.add_variable("farfog", _3ds_float(fog_density + fog_density * 0.5))
                fog_chunk.add_subchunk(fog_color_chunk)
                object_info.add_subchunk(fog_chunk)

            # Add LAYER FOG
            foglayer = next((lk.from_socket.node for lk in ntree if lk.from_socket.node.type == 'VOLUME_SCATTER' and lk.to_socket.node.type in bgshade), False)
            if foglayer:
                layerfog_flag = 0
                if world.mist_settings.falloff == 'QUADRATIC':
                    layerfog_flag |= 0x1
                if world.mist_settings.falloff == 'INVERSE_QUADRATIC':
                    layerfog_flag |= 0x2
                layerfog_chunk = _3ds_chunk(LAYER_FOG)
                layerfog_color_chunk = _3ds_chunk(RGB)
                use_fog_flag = _3ds_chunk(USE_LAYER_FOG)
                layerfog_color_chunk.add_variable("color", _3ds_float_color(foglayer.inputs[0].default_value[:3]))
                layerfog_chunk.add_variable("lowZ", _3ds_float(world.mist_settings.start))
                layerfog_chunk.add_variable("highZ", _3ds_float(world.mist_settings.height))
                layerfog_chunk.add_variable("density", _3ds_float(foglayer.inputs[1].default_value))
                layerfog_chunk.add_variable("flags", _3ds_uint(layerfog_flag))
                layerfog_chunk.add_subchunk(layerfog_color_chunk)
                object_info.add_subchunk(layerfog_chunk)
            if fognode or foglayer and layer.use_pass_mist:
                object_info.add_subchunk(use_fog_flag)
                
        if use_keyframes and world.animation_data or (world.node_tree and world.node_tree.animation_data):
            kfdata.add_subchunk(make_ambient_node(world))

    # Collect translation for transformation matrix
    translation = {}
    rotation = {}
    scale = {}

    # Give all objects a unique ID and build a dictionary from object name to object id
    object_id = {}
    name_id = {}

    for ob, data, matrix in mesh_objects:
        translation[ob.name] = mtx_scale @ ob.location
        rotation[ob.name] = ob.rotation_euler
        scale[ob.name] = mtx_scale.copy()
        name_id[ob.name] = len(name_id)
        object_id[ob.name] = len(object_id)

    for ob in empty_objects:
        translation[ob.name] = mtx_scale @ ob.location
        rotation[ob.name] = ob.rotation_euler
        scale[ob.name] = mtx_scale.copy()
        name_id[ob.name] = len(name_id)

    for ob in light_objects:
        translation[ob.name] = mtx_scale @ ob.location
        rotation[ob.name] = ob.rotation_euler
        scale[ob.name] = mtx_scale.copy()
        name_id[ob.name] = len(name_id)
        object_id[ob.name] = len(object_id)

    for ob in camera_objects:
        translation[ob.name] = mtx_scale @ ob.location
        rotation[ob.name] = ob.rotation_euler
        scale[ob.name] = mtx_scale.copy()
        name_id[ob.name] = len(name_id)
        object_id[ob.name] = len(object_id)

    # Create object chunks for all meshes
    for ob, mesh, matrix in mesh_objects:
        object_chunk = _3ds_chunk(NAMED_OBJECT)

        # Set the object name
        object_chunk.add_variable("name", _3ds_string(sane_name(ob.name)))

        # Make a mesh chunk out of the mesh
        object_chunk.add_subchunk(make_mesh_chunk(ob, mesh, matrix, materialDict, translation))

        # Add hierachy chunk with ID from object_id dictionary
        if use_hierarchy:
            obj_hierarchy_chunk = _3ds_chunk(OBJECT_HIERARCHY)
            obj_hierarchy_chunk.add_variable("hierarchy", _3ds_ushort(object_id[ob.name]))

            # Add parent chunk if object has a parent
            if ob.parent is not None and (ob.parent.name in object_id):
                obj_parent_chunk = _3ds_chunk(OBJECT_PARENT)
                obj_parent_chunk.add_variable("parent", _3ds_ushort(object_id[ob.parent.name]))
                obj_hierarchy_chunk.add_subchunk(obj_parent_chunk)
            object_chunk.add_subchunk(obj_hierarchy_chunk)

        # ensure the mesh has no over sized arrays - skip ones that do!
        # Otherwise we cant write since the array size wont fit in USHORT
        if object_chunk.validate():
            object_info.add_subchunk(object_chunk)
        else:
            operator.report({'WARNING'}, "Object %r can't be written into a .p2 file")

        # Export object node
        if use_keyframes:
            kfdata.add_subchunk(make_object_node(ob, translation, rotation, scale, name_id))

    # Create chunks for all empties - only requires a object node
    if use_keyframes:
        for ob in empty_objects:
            kfdata.add_subchunk(make_object_node(ob, translation, rotation, scale, name_id))

    # Create light object chunks
    for ob in light_objects:
        object_chunk = _3ds_chunk(NAMED_OBJECT)
        obj_light_chunk = _3ds_chunk(OBJECT_LIGHT)
        color_float_chunk = _3ds_chunk(RGB)
        light_distance = translation[ob.name]
        light_attenuate = _3ds_chunk(LIGHT_ATTENUATE)
        light_inner_range = _3ds_chunk(LIGHT_INNER_RANGE)
        light_outer_range = _3ds_chunk(LIGHT_OUTER_RANGE)
        light_energy_factor = _3ds_chunk(LIGHT_MULTIPLIER)
        light_ratio = ob.data.energy if ob.data.type == 'SUN' else ob.data.energy * 0.001
        object_chunk.add_variable("light", _3ds_string(sane_name(ob.name)))
        obj_light_chunk.add_variable("location", _3ds_point_3d(light_distance))
        color_float_chunk.add_variable("color", _3ds_float_color(ob.data.color))
        light_outer_range.add_variable("distance", _3ds_float(ob.data.cutoff_distance))
        light_inner_range.add_variable("radius", _3ds_float(ob.data.shadow_soft_size * 100))
        light_energy_factor.add_variable("energy", _3ds_float(light_ratio))
        obj_light_chunk.add_subchunk(color_float_chunk)
        obj_light_chunk.add_subchunk(light_outer_range)
        obj_light_chunk.add_subchunk(light_inner_range)
        obj_light_chunk.add_subchunk(light_energy_factor)
        if ob.data.use_custom_distance:
            obj_light_chunk.add_subchunk(light_attenuate)

        if ob.data.type == 'SPOT':
            cone_angle = math.degrees(ob.data.spot_size)
            hot_spot = cone_angle - (ob.data.spot_blend * math.floor(cone_angle))
            spot_pos = calc_target(light_distance, rotation[ob.name].x, rotation[ob.name].z)
            spotlight_chunk = _3ds_chunk(LIGHT_SPOTLIGHT)
            spot_roll_chunk = _3ds_chunk(LIGHT_SPOT_ROLL)
            spotlight_chunk.add_variable("target", _3ds_point_3d(spot_pos))
            spotlight_chunk.add_variable("hotspot", _3ds_float(round(hot_spot, 4)))
            spotlight_chunk.add_variable("angle", _3ds_float(round(cone_angle, 4)))
            spot_roll_chunk.add_variable("roll", _3ds_float(round(rotation[ob.name].y, 6)))
            spotlight_chunk.add_subchunk(spot_roll_chunk)
            if ob.data.use_shadow:
                spot_shadow_flag = _3ds_chunk(LIGHT_SPOT_SHADOWED)
                spot_shadow_chunk = _3ds_chunk(LIGHT_SPOT_LSHADOW)
                spot_shadow_chunk.add_variable("bias", _3ds_float(round(ob.data.shadow_buffer_bias,4)))
                spot_shadow_chunk.add_variable("filter", _3ds_float(round((ob.data.shadow_buffer_clip_start * 10),4)))
                spot_shadow_chunk.add_variable("buffer", _3ds_ushort(0x200))
                spotlight_chunk.add_subchunk(spot_shadow_flag)
                spotlight_chunk.add_subchunk(spot_shadow_chunk)
            if ob.data.show_cone:
                spot_cone_chunk = _3ds_chunk(LIGHT_SPOT_SEE_CONE)
                spotlight_chunk.add_subchunk(spot_cone_chunk)
            if ob.data.use_square:
                spot_square_chunk = _3ds_chunk(LIGHT_SPOT_RECTANGLE)
                spotlight_chunk.add_subchunk(spot_square_chunk)
            if ob.scale.x and ob.scale.y != 0.0:
                spot_aspect_chunk = _3ds_chunk(LIGHT_SPOT_ASPECT)
                spot_aspect_chunk.add_variable("aspect", _3ds_float(round((ob.scale.x / ob.scale.y),4)))
                spotlight_chunk.add_subchunk(spot_aspect_chunk)
            if ob.data.use_nodes:
                links = ob.data.node_tree.links
                bptype = 'EMISSION'
                bpmix = 'MIX', 'MIX_RGB', 'EMISSION'
                bptex = 'TEX_IMAGE', 'TEX_ENVIRONMENT'
                bpout = 'ADD_SHADER', 'MIX_SHADER', 'OUTPUT_LIGHT'
                bshade = next((lk.from_node.type for lk in links if lk.from_node.type == bptype and lk.to_node.type in bpout), None)
                bpnode = next((lk.from_node.type for lk in links if lk.from_node.type in bpmix and lk.to_node.type == bshade), bshade)
                bitmap = next((lk.from_node.image for lk in links if lk.from_node.type in bptex and lk.to_node.type == bpnode), False)
                if bitmap and bitmap is not None:
                    spot_projector_chunk = _3ds_chunk(LIGHT_SPOT_PROJECTOR)
                    spot_projector_chunk.add_variable("image", _3ds_string(sane_name(bitmap.name)))
                    spotlight_chunk.add_subchunk(spot_projector_chunk)
            obj_light_chunk.add_subchunk(spotlight_chunk)

        # Add light to object chunk
        object_chunk.add_subchunk(obj_light_chunk)

        # Add hierachy chunks with ID from object_id dictionary
        if use_hierarchy:
            obj_hierarchy_chunk = _3ds_chunk(OBJECT_HIERARCHY)
            obj_parent_chunk = _3ds_chunk(OBJECT_PARENT)
            obj_hierarchy_chunk.add_variable("hierarchy", _3ds_ushort(object_id[ob.name]))
            if ob.parent is not None and (ob.parent.name in object_id):
                obj_parent_chunk = _3ds_chunk(OBJECT_PARENT)
                obj_parent_chunk.add_variable("parent", _3ds_ushort(object_id[ob.parent.name]))
                obj_hierarchy_chunk.add_subchunk(obj_parent_chunk)
            object_chunk.add_subchunk(obj_hierarchy_chunk)

        # Add light object and hierarchy chunks to object info
        object_info.add_subchunk(object_chunk)

        # Export light and spotlight target node
        if use_keyframes:
            kfdata.add_subchunk(make_object_node(ob, translation, rotation, scale, name_id))
            if ob.data.type == 'SPOT':
                kfdata.add_subchunk(make_target_node(ob, translation, rotation, scale, name_id))

    # Create camera object chunks
    for ob in camera_objects:
        object_chunk = _3ds_chunk(NAMED_OBJECT)
        camera_chunk = _3ds_chunk(OBJECT_CAMERA)
        crange_chunk = _3ds_chunk(OBJECT_CAM_RANGES)
        camera_distance = translation[ob.name]
        camera_target = calc_target(camera_distance, rotation[ob.name].x, rotation[ob.name].z)
        object_chunk.add_variable("camera", _3ds_string(sane_name(ob.name)))
        camera_chunk.add_variable("location", _3ds_point_3d(camera_distance))
        camera_chunk.add_variable("target", _3ds_point_3d(camera_target))
        camera_chunk.add_variable("roll", _3ds_float(round(rotation[ob.name].y, 6)))
        camera_chunk.add_variable("lens", _3ds_float(ob.data.lens))
        crange_chunk.add_variable("clipstart", _3ds_float(ob.data.clip_start * 0.1))
        crange_chunk.add_variable("clipend", _3ds_float(ob.data.clip_end * 0.1))
        camera_chunk.add_subchunk(crange_chunk)
        object_chunk.add_subchunk(camera_chunk)

        # Add hierachy chunks with ID from object_id dictionary
        if use_hierarchy:
            obj_hierarchy_chunk = _3ds_chunk(OBJECT_HIERARCHY)
            obj_parent_chunk = _3ds_chunk(OBJECT_PARENT)
            obj_hierarchy_chunk.add_variable("hierarchy", _3ds_ushort(object_id[ob.name]))
            if ob.parent is not None and (ob.parent.name in object_id):
                obj_parent_chunk = _3ds_chunk(OBJECT_PARENT)
                obj_parent_chunk.add_variable("parent", _3ds_ushort(object_id[ob.parent.name]))
                obj_hierarchy_chunk.add_subchunk(obj_parent_chunk)
            object_chunk.add_subchunk(obj_hierarchy_chunk)

        # Add light object and hierarchy chunks to object info
        object_info.add_subchunk(object_chunk)

        # Export camera and target node
        if use_keyframes:
            kfdata.add_subchunk(make_object_node(ob, translation, rotation, scale, name_id))
            kfdata.add_subchunk(make_target_node(ob, translation, rotation, scale, name_id))

    # Add main object info chunk to primary chunk
    primary.add_subchunk(object_info)

    # Add main keyframe data chunk to primary chunk
    if use_keyframes:
        primary.add_subchunk(kfdata)

    # The chunk hierarchy is completely built, now check the size
    primary.get_size()

    # Open the file for writing
    file = open(filepath, 'wb')

    # Recursively write the chunks to file
    primary.write(file)

    # Close the file
    file.close()

    # Clear name mapping vars, could make locals too
    del name_unique[:]
    name_mapping.clear()

    # Debugging only: report the exporting time
    context.window.cursor_set('DEFAULT')
    print(".P2 format export time: %.2f" % (time.time() - duration))

    # Debugging only: dump the chunk hierarchy
    # primary.dump()

    return {'FINISHED'}
