#!/usr/bin/env bash
set -euo pipefail

# Simple reproducible build script for io_scene_p2
# Works when run from:
#  - repo root: ./build.sh
#  - package dir: io_scene_p2/build.sh
#
# Usage:
#   ./build.sh
#   ./build.sh --version 1.2.3
#   ./build.sh --clean

PKG_NAME="io_scene_p2"
EXCLUDES=( "*/.git/*" "*/.gitignore" "*/__pycache__/*" "*.pyc" "*/.DS_Store" "*/.vscode/*" )

# Resolve script location
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# If script lives inside the package directory, set SRC_DIR to script dir.
# Otherwise assume script is at repo root and package dir is under it.
if [[ "$(basename "$SCRIPT_DIR")" == "$PKG_NAME" ]]; then
    SRC_DIR="$SCRIPT_DIR"
    ROOT_DIR="$(dirname "$SCRIPT_DIR")"
else
    ROOT_DIR="$SCRIPT_DIR"
    SRC_DIR="${ROOT_DIR}/${PKG_NAME}"
fi

DIST_DIR="${ROOT_DIR}/dist"
OVERRIDE_VERSION=""

if [[ "$#" -gt 0 ]]; then
    case "$1" in
        --clean)
            echo "Cleaning ${DIST_DIR}..."
            rm -rf "${DIST_DIR}"
            echo "Done."
            exit 0
            ;;
        --version)
            if [[ -z "${2:-}" ]]; then
                echo "Usage: $0 [--clean] [--version X.Y.Z]"
                exit 1
            fi
            OVERRIDE_VERSION="$2"
            ;;
        *)
            echo "Usage: $0 [--clean] [--version X.Y.Z]"
            exit 1
            ;;
    esac
fi

if [[ ! -d "${SRC_DIR}" ]]; then
    echo "ERROR: source directory not found: ${SRC_DIR}" >&2
    exit 2
fi

# Determine version from io_scene_p2/__init__.py by reading bl_info['version']
VERSION="$(python3 - <<PY
import ast,sys
p = "${SRC_DIR}/__init__.py"
try:
    src = open(p, 'r', encoding='utf-8').read()
except FileNotFoundError:
    sys.exit(1)
tree = ast.parse(src)
for node in tree.body:
    if isinstance(node, ast.Assign):
        for target in node.targets:
            if getattr(target, 'id', None) == 'bl_info':
                try:
                    bl = ast.literal_eval(node.value)
                    ver = bl.get('version')
                    if ver:
                        print(".".join(map(str, ver)))
                        sys.exit(0)
                except Exception:
                    pass
print("0.0.0")
PY
)"

if [[ -n "${OVERRIDE_VERSION}" ]]; then
    VERSION="${OVERRIDE_VERSION}"
fi

ZIP_NAME="${PKG_NAME}-${VERSION}.zip"
ZIP_PATH="${DIST_DIR}/${ZIP_NAME}"

mkdir -p "${DIST_DIR}"

# Build exclude args for zip
EXCLUDE_ARGS=()
for ex in "${EXCLUDES[@]}"; do
    EXCLUDE_ARGS+=( -x "$ex" )
done

# Run from ROOT_DIR so zip contains the folder (io_scene_p2/...)
pushd "${ROOT_DIR}" >/dev/null
echo "Creating ${ZIP_PATH} (package folder-in-zip)..."
zip -r -q -9 "${ZIP_PATH}" "${PKG_NAME}" "${EXCLUDE_ARGS[@]}"
popd >/dev/null

echo "Done: ${ZIP_PATH}"
