bl_info = {
    "name": "P-Squared (.p2) Exporter",
    "author": "Pete Pinter",
    "version": (1, 1, 1),
    "blender": (4, 5, 9),  # works for Blender 4.5.x ONLY
    "location": "File > Export > P-Squared (.p2)",
    "description": "Exports Blender meshes and animation actions to .p2 format.",
    "category": "Import-Export",
}

from bpy_extras.io_utils import (
    ExportHelper,
    orientation_helper,
    axis_conversion,
)
from bpy.props import (
    BoolProperty,
    EnumProperty,
    FloatProperty,
    StringProperty,
)
import bpy

if "bpy" in locals():
    import importlib

    if "export_p2" in locals():
        importlib.reload(export_p2)

@orientation_helper(axis_forward='Y', axis_up='Z')

class ExportP2(bpy.types.Operator, ExportHelper):
    
    """Export to P2 file format (.p2)"""
    
    bl_idname = "export_scene.p_squared_p2"
    bl_label = 'Export .p2'
    bl_options = {'PRESET', 'UNDO'}

    filename_ext = ".p2"
    filter_glob: StringProperty(
        default="*.p2",
        options={'HIDDEN'}) # type: ignore

    scale_factor: FloatProperty(
        name="Scale Factor",
        description="Master scale factor for all objects",
        min=0.0, max=100000.0,
        soft_min=0.0, soft_max=100000.0,
        default=1.0,
    ) # type: ignore
    
    use_scene_unit: BoolProperty(
        name="Scene Units",
        description="Take the scene unit length settings into account",
        default=False,
    ) # type: ignore
    
    use_selection: BoolProperty(
        name="Selection",
        description="Export selected objects only",
        default=False,
    ) # type: ignore
    
    object_filter: EnumProperty(
        name="Object Filter", options={'ENUM_FLAG'},
        items=(('WORLD', "World".rjust(11), "", 'WORLD_DATA',0x1),
               ('MESH', "Mesh".rjust(11), "", 'MESH_DATA', 0x2),
               ('LIGHT', "Light".rjust(12), "", 'LIGHT_DATA',0x4),
               ('CAMERA', "Camera".rjust(11), "", 'CAMERA_DATA',0x8),
               ('EMPTY', "Empty".rjust(11), "", 'EMPTY_AXIS',0x10),
               ),
        description="Object types to export",
        default={'WORLD', 'MESH', 'LIGHT', 'CAMERA', 'EMPTY'},
    ) # type: ignore
    
    use_hierarchy: BoolProperty(
        name="Hierarchy",
        description="Export hierarchy chunks",
        default=False,
    ) # type: ignore
    
    use_keyframes: BoolProperty(
        name="Animation",
        description="Write the keyframe data",
        default=True,
    ) # type: ignore
    
    use_cursor: BoolProperty(
        name="Cursor Origin",
        description="Save the 3D cursor location",
        default=False,
    ) # type: ignore

    def execute(self, context):
        from . import export_p2

        keywords = self.as_keywords(ignore=("axis_forward",
                                            "axis_up",
                                            "filter_glob",
                                            "check_existing",
                                            ))
        global_matrix = axis_conversion(to_forward=self.axis_forward,
                                        to_up=self.axis_up,
                                        ).to_4x4()
        keywords["global_matrix"] = global_matrix

        return export_p2.save(self, context, **keywords)

    
    def draw(self, context):
        pass


class P2_PT_export_include(bpy.types.Panel):
    bl_space_type = 'FILE_BROWSER'
    bl_region_type = 'TOOL_PROPS'
    bl_label = "Include"
    bl_parent_id = "FILE_PT_operator"

    @classmethod
    def poll(cls, context):
        sfile = context.space_data
        operator = sfile.active_operator

        return operator.bl_idname == "EXPORT_SCENE_OT_P2"

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True
        layout.use_property_decorate = True

        sfile = context.space_data
        operator = sfile.active_operator

        layrow = layout.row(align=True)
        layrow.prop(operator, "use_selection")
        layrow.label(text="", icon='RESTRICT_SELECT_OFF' if operator.use_selection else 'RESTRICT_SELECT_ON')
        layout.column().prop(operator, "object_filter")
        layrow = layout.row(align=True)
        layrow.prop(operator, "use_hierarchy")
        layrow.label(text="", icon='OUTLINER' if operator.use_hierarchy else 'CON_CHILDOF')
        layrow = layout.row(align=True)
        layrow.prop(operator, "use_keyframes")
        layrow.label(text="", icon='ANIM' if operator.use_keyframes else 'DECORATE_DRIVER')
        layrow = layout.row(align=True)
        layrow.prop(operator, "use_cursor")
        layrow.label(text="", icon='PIVOT_CURSOR' if operator.use_cursor else 'CURSOR')


class P2_PT_export_transform(bpy.types.Panel):
    bl_space_type = 'FILE_BROWSER'
    bl_region_type = 'TOOL_PROPS'
    bl_label = "Transform"
    bl_parent_id = "FILE_PT_operator"

    @classmethod
    def poll(cls, context):
        sfile = context.space_data
        operator = sfile.active_operator

        return operator.bl_idname == "EXPORT_SCENE_OT_P2"

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True
        layout.use_property_decorate = False

        sfile = context.space_data
        operator = sfile.active_operator

        layout.prop(operator, "scale_factor")
        layrow = layout.row(align=True)
        layrow.prop(operator, "use_scene_unit")
        layrow.label(text="", icon='EMPTY_ARROWS' if operator.use_scene_unit else 'EMPTY_DATA')
        layout.prop(operator, "axis_forward")
        layout.prop(operator, "axis_up")


# Add to a menu
def menu_func_export(self, context):
    self.layout.operator(ExportP2.bl_idname, text="P-Squared (.p2)")

def register():
    import bpy
    min_version = (4, 0, 0)
    max_version = (4, 9, 9)
    if not (min_version <= bpy.app.version <= max_version):
        raise Exception(
            f"P2 Exporter supports Blender {min_version[0]}.{min_version[1]}–"
            f"{max_version[0]}.{max_version[1]} only. "
            f"You’re running Blender {bpy.app.version_string}."
        )

    bpy.utils.register_class(ExportP2)
    bpy.utils.register_class(P2_PT_export_include)
    bpy.utils.register_class(P2_PT_export_transform)
    bpy.types.TOPBAR_MT_file_export.append(menu_func_export)


def unregister():
    bpy.utils.unregister_class(ExportP2)
    bpy.utils.unregister_class(P2_PT_export_include)
    bpy.utils.unregister_class(P2_PT_export_transform)
    bpy.types.TOPBAR_MT_file_export.remove(menu_func_export)


if __name__ == "__main__":
    register()
